import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import AboutClient from "./AboutClient";
import { Metadata } from "next";

export const dynamic = "force-dynamic";

function stripHtml(html: string): string {
    if (!html) return "";
    return html.replace(/<[^>]*>?/gm, "").trim();
}

interface CounterItem {
    img: string;
    value: number;
    suffix: string;
    text: string;
}

import { buildPageMeta } from "@/lib/seo/generalMeta";

export async function generateMetadata(): Promise<Metadata> {
    const res = await API_ENDPOINTS.PAGE(2);

    if (!res.ok) {
        return {
            title: "About Us",
            description: "Learn more about our company",
        };
    }

    const data = await res.json();
    return buildPageMeta(data, "/about", "About Us");
}

async function fetchAboutData() {
    const [
        pageRes,
        aboutRes,
        counterRes,
        visionRes,
        keyBenefitsRes,
        companyRes,
        testimonialsSectionRes,
        testimonialsRes,
    ] = await Promise.all([
        API_ENDPOINTS.PAGE(2),
        API_ENDPOINTS.SECTION(27),
        API_ENDPOINTS.SUBSECTIONS(2, 27),
        API_ENDPOINTS.SUBSECTIONS(2, 28),
        API_ENDPOINTS.SUBSECTIONS(1, 57),
        API_ENDPOINTS.COMPANY_CONTACT(),
        API_ENDPOINTS.SECTION(55),
        API_ENDPOINTS.TESTIMONIALS(),
    ]);

    if (
        !pageRes.ok ||
        !aboutRes.ok ||
        !counterRes.ok ||
        !visionRes.ok ||
        !keyBenefitsRes.ok ||
        !companyRes.ok ||
        !testimonialsSectionRes.ok ||
        !testimonialsRes.ok
    ) {
        throw new Error("Failed to load About page data");
    }

    const pageData = await pageRes.json();
    const aboutSectionData = await aboutRes.json();
    const counterSectionData = await counterRes.json();
    const visionApiData = await visionRes.json();
    const keyBenefitsApi = await keyBenefitsRes.json();
    const companyApi = await companyRes.json();
    const testimonialsSectionData = await testimonialsSectionRes.json();
    const testimonialsList = await testimonialsRes.json();

    const counterItems: CounterItem[] =
        (counterSectionData.subsections || []).map((item: any) => {
            const title = item.title || "";
            const valueMatch = title.match(/\d+/);
            const suffixMatch = title.replace(/\d+/g, "").trim();

            return {
                img: item.icon,
                value: valueMatch ? Number(valueMatch[0]) : 0,
                suffix: suffixMatch,
                text: stripHtml(item.desc),
            };
        });

    return {
        breadcrumbHeadline: pageData.breadcrumb_headline,
        aboutData: aboutSectionData,
        counterData: counterItems,
        visionData: {
            parent: {
                headline: visionApiData.parent_section_headline,
                image: visionApiData.parent_section_image,
            },
            subsections: visionApiData.subsections || [],
        },
        keyBenefitsData: {
            parent: {
                headline: keyBenefitsApi.parent_section_headline,
                description: keyBenefitsApi.parent_section_description,
            },
            subsections: keyBenefitsApi.subsections || [],
            company: companyApi,
        },
        testimonialsData: {
            headline: testimonialsSectionData.section_headline,
            description: testimonialsSectionData.description,
            testimonials: testimonialsList || [],
        },
    };
}

export default async function AboutPage() {
    const data = await fetchAboutData();
    return <AboutClient {...data} />;
}
