"use client";

import React, { useEffect, useState, useRef } from "react";
import { API_ENDPOINTS, getToken } from "@/lib/apiEndpoints";
import InnerBanner from "../../../components/InnerBanner";
import { useRouter } from "next/navigation";
import { useCart } from "@/context/CartContext";
import BuyNowGate from "@/components/BuyNowGate";

interface HardCopyAssignmentClientProps {
    pageData: any;
    sectionDescription: any;
    categoriesTree: any[];
    allLanguages: any[];
    allSessions: any[];
}

function stripHtml(html: string): string {
    if (!html) return "";
    return html.replace(/<[^>]*>?/gm, "").trim();
}

export default function HardCopyAssignmentClient({
    pageData,
    sectionDescription,
    categoriesTree,
    allLanguages,
    allSessions
}: HardCopyAssignmentClientProps) {
    const router = useRouter();
    const [isLoggedIn, setIsLoggedIn] = useState<boolean>(false); // Initial false to avoid hydration mismatch

    const { setBuyNowItem } = useCart();
    const [submitting, setSubmitting] = useState(false);

    // Initial Data from props
    const breadcrumbHeadline = pageData?.breadcrumb_headline || "";
    const headline = pageData?.page_headline || "";
    const description = pageData?.page_description ? stripHtml(pageData.page_description) : "";
    const section = sectionDescription || "";

    // Cascading selections (encrypted IDs)
    const [selectedCategory, setSelectedCategory] = useState<string>("");
    const [selectedCourse, setSelectedCourse] = useState<string>("");
    const [selectedSubjects, setSelectedSubjects] = useState<{ encrypted_id: string; subject_code: string; }[]>([]);
    const [isTypingSubjects, setIsTypingSubjects] = useState(false);
    const [manualSubjects, setManualSubjects] = useState<string>("");
    const [selectionError, setSelectionError] = useState<string | null>(null);

    const [mediums, setMediums] = useState<any[]>([]);
    const [sessions, setSessions] = useState<any[]>([]);

    // ---------- 🔄 Sync login state across tabs ----------
    useEffect(() => {
        setIsLoggedIn(!!getToken()); // Set on mount

        const syncAuthState = () => {
            setIsLoggedIn(!!getToken());
        };

        const handleVisibilityChange = () => {
            if (!document.hidden) {
                syncAuthState();
            }
        };

        const handleFocus = () => {
            syncAuthState();
        };

        document.addEventListener("visibilitychange", handleVisibilityChange);
        window.addEventListener("focus", handleFocus);

        return () => {
            document.removeEventListener("visibilitychange", handleVisibilityChange);
            window.removeEventListener("focus", handleFocus);
        };
    }, []);

    // ---------- Reset dependent selections ----------
    useEffect(() => {
        setSelectedCourse("");
        setSelectedSubjects([]);
        setMediums([]);
        setSessions([]);
    }, [selectedCategory]);

    useEffect(() => {
        const course = getSelectedCourse();

        if (!course || !Array.isArray(course.subjects) || course.subjects.length === 0) {
            setMediums([]);
            setSessions([]);
            return;
        }

        setMediums(allLanguages || []);
        setSessions(allSessions || []);
    }, [selectedCourse, allLanguages, allSessions]);

    useEffect(() => {
        if (isTypingSubjects) return;

        const allSubjectCodes = getSubjects().map(
            (s: any) => s.subject_code.toUpperCase()
        );

        const selectedCodes = selectedSubjects.map(
            s => s.subject_code.toUpperCase()
        );

        const typedCodes = manualSubjects
            .toUpperCase()
            .split(",")
            .map(s => s.trim())
            .filter(Boolean);

        const filteredCodes = typedCodes.filter(code =>
            !allSubjectCodes.includes(code) || selectedCodes.includes(code)
        );

        const finalCodes = Array.from(
            new Set([...filteredCodes, ...selectedCodes])
        );

        setManualSubjects(finalCodes.join(", "));
    }, [selectedSubjects]);

    // ---------- Helpers ----------
    const getSelectedCategory = () =>
        categoriesTree.find((c) => c.id === selectedCategory);

    const getSelectedCourse = () =>
        getSelectedCategory()?.courses?.find(
            (c: any) => c.encrypted_id === selectedCourse
        );

    const getSubjects = () =>
        getSelectedCourse()?.subjects || [];

    // ---------- Handle "Select All" toggle ----------
    const handleSelectAllSubjects = () => {
        const subjects = getSubjects();

        if (isAllSelected) {
            setSelectedSubjects([]);
        } else {
            setSelectedSubjects(
                subjects.map((s: any) => ({
                    encrypted_id: s.encrypted_id,
                    subject_code: s.subject_code,
                }))
            );
        }
    };

    // ---------- Handle manual subjects selection ----------
    const handleManualSubjectsChange = (value: string) => {
        setIsTypingSubjects(true);

        const upperValue = value.toUpperCase();
        setManualSubjects(upperValue);

        const typedCodes = upperValue
            .split(",")
            .map(s => s.trim())
            .filter(Boolean);

        const allSubjects = getSubjects();

        const matchedSubjects = allSubjects.filter((sub: any) =>
            typedCodes.includes(sub.subject_code.toUpperCase())
        );

        setSelectedSubjects(
            matchedSubjects.map((sub: any) => ({
                encrypted_id: sub.encrypted_id,
                subject_code: sub.subject_code,
            }))
        );

        // Allow effect to run AFTER typing settles
        setTimeout(() => setIsTypingSubjects(false), 0);
    };

    // ---------- Handle individual subject selection ----------
    const handleSubjectChange = (subject: any) => {
        setSelectedSubjects((prev) => {
            const exists = prev.find(
                (s) => s.encrypted_id === subject.encrypted_id
            );

            if (exists) {
                return prev.filter(
                    (s) => s.encrypted_id !== subject.encrypted_id
                );
            }

            return [
                ...prev,
                {
                    encrypted_id: subject.encrypted_id,
                    subject_code: subject.subject_code,
                },
            ];
        });
    };

    // Check if all subjects are selected
    const allSubjects = getSubjects().filter((sub: any) => sub?.encrypted_id);
    const isAllSelected = allSubjects.length > 0 && selectedSubjects.length === allSubjects.length;

    const formRef = useRef<HTMLFormElement | null>(null);

    const handleBuyNow = async (e: React.FormEvent<HTMLFormElement>) => {
        e.preventDefault();

        if (!formRef.current) return;
        if (submitting) return;

        setSelectionError(null);
        setSubmitting(true);

        try {
            // Auth check (live)
            const token = getToken();
            if (!token) {
                setIsLoggedIn(false);
                router.push("/login");
                return;
            }

            if (!selectedCourse || selectedSubjects.length === 0) {
                setSelectionError("Please select Course and Subjects.");
                return;
            }

            const course = getSelectedCourse();

            if (!course) {
                setSelectionError("Invalid course selection.");
                return;
            }

            // From checkboxes
            const selectedCodes = selectedSubjects.map(s => s.subject_code);

            // From manual input
            const manualCodes = manualSubjects
                .split(",")
                .map(s => s.trim())
                .filter(Boolean);

            // Merge + dedupe
            const finalSubjectCodes = Array.from(
                new Set([...selectedCodes, ...manualCodes])
            );

            const subjectCodesString = finalSubjectCodes.join(", ");

            const formData = new FormData(formRef.current);

            formData.append("degree_cat_gen_id", selectedCategory);
            formData.append("course", selectedCourse);
            formData.append("subjects", subjectCodesString);
            formData.append("assignment_method", "handwritten-hard-copy");
            formData.append("price", String(400));
            formData.append("quantity", String(finalSubjectCodes.length ?? 1));

            const saveRes = await API_ENDPOINTS.HANDWRITTEN.SAVE(formData);

            if (!saveRes.ok) {
                const err = await saveRes.json();
                throw new Error(err.message || "Failed to save data");
            }

            const saved = await saveRes.json();
            const rawString = `${Date.now()}`;

            setBuyNowItem({
                id: `handwritten-hard-copy-${rawString}`,
                name: subjectCodesString,
                price: 400,
                qty: finalSubjectCodes.length ?? 1,
                img: "",
                courseCode: course.course_url,
                itemCode: "handwritten-hard-copy",
                subjectIds: finalSubjectCodes,
                selectedType: null,
                selectedMedium: null,
                selectedSession: null,
                dataSaveId: null,
                pageName: "hard-copy-page"
            });

            router.push("/checkout");
        } catch (err: any) {
            console.error("❌ Save failed:", err);
            setSelectionError(err.message || "Something went wrong. Please try again.");
        } finally {
            setSubmitting(false);
        }
    };

    return (
        <>
            <style jsx global>{`
                body {
                    font-family: 'Plus Jakarta Sans', sans-serif;
                }
                .form-floating > .form-control,
                .form-floating > .form-select {
                    border: 1px solid #e2e8f0;
                    border-radius: 12px;
                    background-color: #f8fafc;
                    transition: all 0.2s ease;
                }
                .form-floating > .form-control:focus,
                .form-floating > .form-select:focus {
                    background-color: #fff;
                    border-color: #3b82f6;
                    box-shadow: 0 0 0 4px rgba(59, 130, 246, 0.1);
                }
                .section-title {
                    font-size: 1.1rem;
                    font-weight: 700;
                    color: #1e293b;
                    margin-bottom: 1.25rem;
                    display: flex;
                    align-items: center;
                    gap: 0.75rem;
                }
                .section-title::before {
                    content: '';
                    display: block;
                    width: 4px;
                    height: 24px;
                    background: linear-gradient(to bottom, #3b82f6, #60a5fa);
                    border-radius: 4px;
                }
                .submit-btn {
                    background: linear-gradient(135deg, #2563eb, #1d4ed8);
                    border: none;
                    border-radius: 12px;
                    padding: 1rem 2rem;
                    font-weight: 600;
                    letter-spacing: 0.5px;
                    box-shadow: 0 10px 15px -3px rgba(37, 99, 235, 0.2);
                    transition: all 0.3s ease;
                }
                .submit-btn:hover {
                    transform: translateY(-2px);
                    box-shadow: 0 20px 25px -5px rgba(37, 99, 235, 0.3);
                }
                .main-card {
                    border: none;
                    border-radius: 24px;
                    box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.05); 
                    background: white;
                    padding: 3rem;
                }
                .custom-bullets {
                    list-style: none;
                    padding-left: 0;
                    max-width: 750px;
                }

                .custom-bullets li {
                    margin-bottom: 10px;
                }

                .custom-bullets li::before {
                    content: "📌 ";
                    color: #0d6efd;
                    font-weight: bold;
                }

                a {
                    text-decoration: none;
                    color: #000;
                }

                a:hover {
                    color: #0d6efd;
                }

                @media (max-width: 768px) {
                    .main-card {
                        padding: 1.5rem;
                    }
                }

                .subject-multiselect-wrapper {
                    border: 2px solid #dee2e6;
                    border-radius: 0.5rem;
                    background-color: #fff;
                    transition: all 0.3s ease;
                }

                .subject-multiselect-wrapper:hover {
                    border-color: #0d6efd;
                    box-shadow: 0 0 0 0.2rem rgba(13, 110, 253, 0.1);
                }

                .subject-multiselect-wrapper.disabled {
                    background-color: #f8f9fa;
                    opacity: 0.6;
                    cursor: not-allowed;
                }

                .subject-scroll-area {
                    max-height: 280px;
                    overflow-y: auto;
                    padding: 0.75rem;
                }

                .subject-scroll-area::-webkit-scrollbar {
                    width: 6px;
                }

                .subject-scroll-area::-webkit-scrollbar-track {
                    background: #f1f1f1;
                    border-radius: 10px;
                }

                .subject-scroll-area::-webkit-scrollbar-thumb {
                    background: #cbd5e0;
                    border-radius: 10px;
                }

                .subject-scroll-area::-webkit-scrollbar-thumb:hover {
                    background: #a0aec0;
                }

                .select-all-section {
                    padding: 0.75rem;
                    background: linear-gradient(135deg, #4a90ff 0%, #3081ff 100%);
                    border-radius: 0.375rem;
                    margin-bottom: 0.75rem;
                }

                .select-all-section label {
                    color: white;
                    font-weight: 600;
                    margin-bottom: 0;
                    cursor: pointer;
                    user-select: none;
                }

                .select-all-section .form-check-input {
                    width: 1.25rem;
                    height: 1.25rem;
                    cursor: pointer;
                    border: 2px solid white;
                    background-color: rgba(255, 255, 255, 0.2);
                }

                .select-all-section .form-check-input:checked {
                    background-color: white;
                    border-color: white;
                }

                .subject-item {
                    padding: 0.625rem 0.75rem;
                    margin-bottom: 0.5rem;
                    border-radius: 0.375rem;
                    transition: all 0.2s ease;
                    border: 1px solid transparent;
                }

                .subject-item:hover {
                    background-color: #f8f9fa;
                    border-color: #e9ecef;
                }

                .subject-item .form-check-input {
                    width: 1.15rem;
                    height: 1.15rem;
                    cursor: pointer;
                    margin-top: 0.15rem;
                }

                .subject-item .form-check-label {
                    cursor: pointer;
                    margin-left: 0.5rem;
                    user-select: none;
                    font-size: 0.95rem;
                    color: #495057;
                }

                .subject-item .form-check-input:checked {
                    background-color: #198754;
                    border-color: #198754;
                }

                .subject-item.checked {
                    background-color: #d1e7dd;
                    border-color: #198754;
                }

                .selected-badge {
                    display: inline-flex;
                    align-items: center;
                    gap: 0.5rem;
                    padding: 0.5rem 1rem;
                    background: linear-gradient(135deg, #4a90ff 0%, #3081ff 100%);
                    color: white;
                    border-radius: 2rem;
                    font-size: 0.875rem;
                    font-weight: 600;
                    margin-top: 0.75rem;
                }

                .selected-badge i {
                    font-size: 1rem;
                }

                .empty-state {
                    text-align: center;
                    padding: 3rem 1rem;
                    color: #6c757d;
                }

                .empty-state i {
                    font-size: 3.5rem;
                    opacity: 0.2;
                    margin-bottom: 1rem;
                    display: block;
                }

                @keyframes fadeIn {
                    from {
                        opacity: 0;
                        transform: translateY(-10px);
                    }
                    to {
                        opacity: 1;
                        transform: translateY(0);
                    }
                }

                .subject-item {
                    animation: fadeIn 0.3s ease;
                }
                
            `}</style>
            <InnerBanner title={breadcrumbHeadline} breadcrumb={breadcrumbHeadline} />

            <div className="py-5" style={{ background: "#f1f5f9", minHeight: "100vh" }}>
                <div className="container">

                    <div className="hard-copy-heading mb-5">
                        <h2>{headline}</h2>
                        <p dangerouslySetInnerHTML={{ __html: description }} />
                    </div>

                    <div className="row justify-content-center">
                        <div className="col-xl-12 col-lg-12">
                            <div className="main-card">

                                {/* ✅ REPLACED BLOCK */}
                                <BuyNowGate
                                    isLoggedIn={isLoggedIn}
                                    handleBuyNow={handleBuyNow}
                                    variant="login-block"
                                />
                                {isLoggedIn && (
                                    <form ref={formRef} onSubmit={handleBuyNow}>
                                        {/* ================= Academic Details ================= */}
                                        <div className="mb-5">
                                            <h4 className="section-title">Academic Details</h4>

                                            <div className="row g-4">

                                                {/* Category */}
                                                <div className="col-md-6">
                                                    <div className="form-floating">
                                                        <select
                                                            className="form-select"
                                                            value={selectedCategory}
                                                            onChange={(e) => {
                                                                setSelectedCategory(e.target.value);
                                                                setSelectedCourse("");
                                                                setSelectedSubjects([]);
                                                            }}
                                                        >
                                                            <option value="">-- Select Degree --</option>
                                                            {categoriesTree.map((deg) => (
                                                                <option key={deg.id} value={deg.id}>
                                                                    {deg.label}
                                                                </option>
                                                            ))}
                                                        </select>
                                                        <label htmlFor="degreeProgram">Degree Program</label>
                                                    </div>
                                                </div>

                                                {/* Course */}
                                                <div className="col-md-6">
                                                    <div className="form-floating">
                                                        <select
                                                            className="form-select"
                                                            value={selectedCourse}
                                                            disabled={!selectedCategory}
                                                            onChange={(e) => {
                                                                setSelectedCourse(e.target.value);
                                                                setSelectedSubjects([]);
                                                            }}
                                                        >
                                                            <option value="">-- Select Course --</option>
                                                            {(getSelectedCategory()?.courses || []).map((course: any) => (
                                                                <option key={course.encrypted_id} value={course.encrypted_id}>
                                                                    {course.code} - {course.title}
                                                                </option>
                                                            ))}
                                                        </select>
                                                        <label htmlFor="course">Course</label>
                                                    </div>
                                                </div>

                                                {/* Subject */}
                                                <div className="col-12">
                                                    <div
                                                        className={`subject-multiselect-wrapper ${!selectedCourse ? 'disabled' : ''}`}
                                                    >
                                                        {!selectedCourse ? (
                                                            <div className="empty-state">
                                                                <i className="fas fa-book"></i>
                                                                <p className="mb-0 fw-semibold">Please select a Course first for Subject Codes</p>
                                                            </div>
                                                        ) : getSubjects().length === 0 ? (
                                                            <div className="empty-state">
                                                                <i className="fas fa-inbox"></i>
                                                                <p className="mb-0 fw-semibold">No subjects available</p>
                                                            </div>
                                                        ) : (
                                                            <div className="subject-scroll-area">
                                                                {/* Select All */}
                                                                <div className="select-all-section">
                                                                    <div className="form-check">
                                                                        <input
                                                                            className="form-check-input"
                                                                            type="checkbox"
                                                                            id="selectAllSubjects"
                                                                            checked={isAllSelected}
                                                                            onChange={handleSelectAllSubjects}
                                                                        />
                                                                        <label
                                                                            className="form-check-label"
                                                                            htmlFor="selectAllSubjects"
                                                                        >
                                                                            Select All Subjects ({allSubjects.length})
                                                                        </label>
                                                                    </div>
                                                                </div>

                                                                {/* Individual Subjects */}
                                                                {getSubjects().map((sub: any) => {
                                                                    if (!sub?.encrypted_id) return null;

                                                                    const isChecked = selectedSubjects.some(
                                                                        (s) => s.encrypted_id === sub.encrypted_id
                                                                    );

                                                                    return (
                                                                        <div
                                                                            key={sub.encrypted_id}
                                                                            className={`subject-item ${isChecked ? 'checked' : ''}`}
                                                                        >
                                                                            <div className="form-check">
                                                                                <input
                                                                                    className="form-check-input"
                                                                                    type="checkbox"
                                                                                    id={`subject-${sub.encrypted_id}`}
                                                                                    checked={isChecked}
                                                                                    onChange={() => handleSubjectChange(sub)}
                                                                                />
                                                                                <label
                                                                                    className="form-check-label"
                                                                                    htmlFor={`subject-${sub.encrypted_id}`}
                                                                                >
                                                                                    {sub.subject_code}
                                                                                </label>
                                                                            </div>
                                                                        </div>
                                                                    );
                                                                })}
                                                            </div>
                                                        )}
                                                    </div>
                                                    {selectedSubjects.length > 0 && (
                                                        <div className="selected-badge">
                                                            <i className="fas fa-check-circle"></i>
                                                            <span>
                                                                {selectedSubjects.length} Subject{selectedSubjects.length !== 1 ? 's' : ''} Selected
                                                            </span>
                                                        </div>
                                                    )}
                                                </div>

                                                {/* Editable Selected Subjects */}
                                                <div className="col-12 mt-4">
                                                    <div className="form-floating">
                                                        <textarea
                                                            className="form-control"
                                                            id="editableSubjects"
                                                            placeholder="Enter subject codes"
                                                            style={{ height: "100px" }}
                                                            value={manualSubjects}
                                                            onChange={(e) => handleManualSubjectsChange(e.target.value)}
                                                        />
                                                        <label htmlFor="editableSubjects">
                                                            Selected / Additional Subject Codes (Comma Separated)
                                                        </label>
                                                    </div>
                                                    <small className="text-muted ms-1">
                                                        You can add or edit subject codes manually (e.g. ECO101, MTH203)
                                                    </small>
                                                </div>

                                                {/* Medium */}
                                                <div className="col-md-6">
                                                    <div className="form-floating">
                                                        <select className="form-select" name="language" id="assignmentMedium" disabled={!selectedCourse}>
                                                            <option value="">-- Select Medium --</option>
                                                            {(mediums || []).map((m: any) => (
                                                                <option key={m.encrypted_id} value={m.encrypted_id}>
                                                                    {m.language_name}
                                                                </option>
                                                            ))}
                                                        </select>
                                                        <label htmlFor="assignmentMedium">Medium</label>
                                                    </div>
                                                </div>

                                                {/* Session */}
                                                <div className="col-md-6">
                                                    <div className="form-floating">
                                                        <select className="form-select" name="session" id="session" disabled={!selectedCourse}>
                                                            <option value="">-- Select Session --</option>
                                                            {(sessions || []).map((s: any) => (
                                                                <option key={s.encrypted_id} value={s.encrypted_id}>
                                                                    {s.session_name}
                                                                </option>
                                                            ))}
                                                        </select>
                                                        <label htmlFor="session">Session</label>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>

                                        <hr className="my-5" style={{ opacity: 0.1 }} />

                                        {/* ================= Submission Preferences ================= */}
                                        <div className="mb-5">
                                            <h4 className="section-title">Submission Preferences</h4>
                                            <div className="row g-4">
                                                <div className="col-12">
                                                    <div className="form-floating">
                                                        <select className="form-select" name="assignment_method" id="method" defaultValue="handwritten-hard-copy">
                                                            <option value="handwritten-hard-copy">
                                                                Handwritten (Hard Copy Only) - Rs 400/- per Assignment
                                                            </option>
                                                        </select>
                                                        <label htmlFor="method">Assignment Method</label>
                                                    </div>
                                                </div>

                                                <div className="col-12">
                                                    <label className="form-label text-muted small fw-bold ms-1">
                                                        Upload Question Paper (PDF)
                                                    </label>
                                                    <div
                                                        className="p-4 border-2 border-dashed rounded-4 text-center"
                                                        style={{
                                                            backgroundColor: "#f8fafc",
                                                            borderColor: "#e2e8f0",
                                                            borderStyle: "dashed",
                                                        }}
                                                    >
                                                        <i className="fa-solid fa-cloud-arrow-up text-primary fs-2 mb-3"></i>
                                                        <input
                                                            type="file"
                                                            name="question_paper_file"
                                                            className="form-control visually-hidden"
                                                            id="fileUpload"
                                                            accept=".pdf"
                                                        />
                                                        <label
                                                            htmlFor="fileUpload"
                                                            className="d-block btn btn-outline-primary btn-sm mx-auto"
                                                            style={{ width: "fit-content" }}
                                                        >
                                                            Choose File
                                                        </label>
                                                        <small className="d-block text-muted mt-2">
                                                            Max file size: 5MB
                                                        </small>
                                                    </div>
                                                </div>

                                                <div className="col-12">
                                                    <div className="form-floating">
                                                        <textarea
                                                            className="form-control"
                                                            name="comments"
                                                            placeholder="Comments"
                                                            id="comments"
                                                            style={{ height: "100px" }}
                                                        ></textarea>
                                                        <label htmlFor="comments">
                                                            Special Instructions / Comments
                                                        </label>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>

                                        {selectionError && (
                                            <div className="alert alert-danger mt-3">
                                                {selectionError}
                                            </div>
                                        )}

                                        <div className="d-grid">
                                            <button type="submit" className="submit-btn text-white">
                                                {submitting ? (
                                                    <>
                                                        <span className="spinner-border spinner-border-sm"></span> Saving...
                                                    </>
                                                ) : (
                                                    "Buy Now"
                                                )}
                                            </button>
                                        </div>

                                    </form>
                                )}
                            </div>
                        </div>
                    </div>

                    <section className="bg-light mt-5">
                        <div className="container py-3">
                            <div className="row">
                                <div className="col-sm-12 align-self-center text-justify" dangerouslySetInnerHTML={{ __html: section }} />
                            </div>
                        </div>
                    </section>

                </div>
            </div>
        </>
    );
}
