import React from 'react';
import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import SolvedAssignmentsClient from "./SolvedAssignmentsClient";
import { buildAssignmentListMeta } from "@/lib/seo/assignmentMeta";

export const dynamic = "force-dynamic";

async function fetchSolvedAssignmentsData() {
    try {
        const [res, res1] = await Promise.all([
            API_ENDPOINTS.PAGE(3),
            API_ENDPOINTS.ASSIGNMENTS.CATEGORIES_COURSES("solved"),
        ]);

        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        if (!res1.ok) throw new Error(`HTTP ${res1.status}`);

        const [pageData, categoriesData] = await Promise.all([
            res.json(),
            res1.json(),
        ]);

        return { pageData, categoriesData };
    } catch (err: any) {
        console.error("❌ Failed to load solved assignments data", err);
        return null;
    }
}

export async function generateMetadata(): Promise<Metadata> {
    const data = await fetchSolvedAssignmentsData();
    if (!data || !data.pageData) return { title: "Solved Assignments" };

    const meta = buildAssignmentListMeta(data.pageData, "Solved Assignments | IGNOU");
    return {
        ...meta,
        alternates: {
            canonical: "/assignments/solved",
        }
    };
}

export default async function SolvedAssignmentsPage() {
    const data = await fetchSolvedAssignmentsData();

    if (!data) {
        return <div className="text-danger text-center py-5">Failed to load solved assignments.</div>;
    }

    return <SolvedAssignmentsClient pageData={data.pageData} categoriesData={data.categoriesData} />;
}
