import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import CoverPageClient from "./CoverPageClient";
import { buildPageMeta } from "@/lib/seo/generalMeta";

export const dynamic = "force-dynamic";

async function fetchCoverPageData() {
    try {
        const [res, res2, cover_page_res] = await Promise.all([
            API_ENDPOINTS.PAGE(14),
            API_ENDPOINTS.SECTION(96),
            API_ENDPOINTS.COVER_PAGE(),
        ]);

        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        if (!res2.ok) throw new Error(`HTTP ${res2.status}`);
        if (!cover_page_res.ok) throw new Error(`HTTP ${cover_page_res.status}`);

        const [pageData, sectionData, coverPageData] = await Promise.all([
            res.json(),
            res2.json(),
            cover_page_res.json(),
        ]);

        return {
            pageData,
            sectionData,
            coverPage: coverPageData || { top_banner_files: [], files: [] },
        };
    } catch (err: any) {
        console.error("❌ Failed to load free download data", err);
        return null;
    }
}

export async function generateMetadata(): Promise<Metadata> {
    const data = await fetchCoverPageData();
    if (!data || !data.pageData) return { title: "Free Downloads" };
    return buildPageMeta(data.pageData, "/free-download", "Free Downloads");
}

export default async function CoverPagePage() {
    const data = await fetchCoverPageData();

    if (!data) {
        return <div className="text-danger text-center py-5">Failed to load free downloads.</div>;
    }

    return (
        <CoverPageClient
            pageData={data.pageData}
            sectionData={data.sectionData}
            coverPage={data.coverPage}
        />
    );
}
