import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import FaqClient from "./FaqClient";
import { buildPageMeta } from "@/lib/seo/generalMeta";

export const dynamic = "force-dynamic";

async function fetchFaqData() {
    try {
        const [pageRes, faqsRes] = await Promise.all([
            API_ENDPOINTS.PAGE(6),
            API_ENDPOINTS.FAQS(),
        ]);

        if (!pageRes.ok) throw new Error(`PAGE HTTP ${pageRes.status}`);
        if (!faqsRes.ok) throw new Error(`FAQS HTTP ${faqsRes.status}`);

        const [pageData, faqsData] = await Promise.all([
            pageRes.json(),
            faqsRes.json(),
        ]);

        return {
            pageData,
            faqsData: faqsData || [],
        };
    } catch (err: any) {
        console.error('❌ Failed to load FAQ data', err);
        return null;
    }
}

export async function generateMetadata(): Promise<Metadata> {
    const data = await fetchFaqData();
    if (!data || !data.pageData) return { title: "FAQ" };
    return buildPageMeta(data.pageData, "/faqs", "FAQ");
}

export default async function FAQPage() {
    const data = await fetchFaqData();

    if (!data) {
        return <div className="text-danger text-center py-5">Failed to load FAQs.</div>;
    }

    return (
        <FaqClient
            pageData={data.pageData}
            faqs={data.faqsData}
        />
    );
}
