import type { Metadata } from "next";
import { Geist, Geist_Mono } from "next/font/google";
import "./globals.css";

import Navbar from "../components/Navbar";
import Footer from "../components/Footer";
import { CartProvider } from "../context/CartContext";
import CartDrawer from "../components/CartDrawer";
import Script from "next/script";
import { getSiteStatus } from "@/lib/siteStatus";
import { API_ENDPOINTS } from "@/lib/apiEndpoints";

const geistSans = Geist({
    variable: "--font-geist-sans",
    subsets: ["latin"],
});

const geistMono = Geist_Mono({
    variable: "--font-geist-mono",
    subsets: ["latin"],
});

export const metadata: Metadata = {
    metadataBase: new URL(process.env.NEXT_PUBLIC_SITE_URL || 'http://localhost:3000'),
    title: {
        default: "IGNOU",
        template: "%s",
    },
    description: "India’s best assignment help platform",
};

async function getWebsiteData() {
    try {
        const [companyRes] = await Promise.all([
            API_ENDPOINTS.COMPANY_CONTACT(),
        ]);

        if (!companyRes.ok) {
            throw new Error("Failed to fetch website data");
        }

        const companyData = await companyRes.json();

        return {
            company: companyData,
        };

    } catch (error) {
        console.error("Footer API Error:", error);
        return null;
    }
}

export default async function RootLayout({
    children,
}: Readonly<{
    children: React.ReactNode;
}>) {
    const siteStatus = await getSiteStatus();
    const websiteData = await getWebsiteData();

    let content;

    if (!siteStatus.site_active && !siteStatus.maintenance_mode) {
        content = (
            <div className="container text-center mt-5">
                <h1>Site is currently offline</h1>
                <p>Please check back later.</p>
            </div>
        );
    } else if (siteStatus.maintenance_mode) {
        content = (
            <div className="container text-center mt-5">
                <h1>Maintenance in progress</h1>
                <p>We’ll be back shortly.</p>
            </div>
        );
    } else {
        content = (
            <>
                {/* <Navbar />
                <main>{children}</main>
                <Footer /> */}
                <CartProvider>
                    <Navbar data={websiteData} />
                    <CartDrawer />
                    <main>{children}</main>
                    <Footer data={websiteData} />
                </CartProvider>
                <Script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js" strategy="afterInteractive" />
            </>
        );
    }

    return (
        <html lang="en">
            <head>
                {/* Bootstrap CSS */}
                <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet" />

                {/* Font Awesome Icons */}
                <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet" />

                {/* Preload critical fonts */}
                <link rel="preload" href="/font/Inter24pt-Regular.woff2" as="font" type="font/woff2" crossOrigin="anonymous" />
                <link rel="preload" href="/font/Inter24pt-Bold.woff2" as="font" type="font/woff2" crossOrigin="anonymous" />
            </head>
            <body className={`${geistSans.variable} ${geistMono.variable}`} suppressHydrationWarning={true}>
                {content}
            </body>
        </html>
    );
}
