import React, { Suspense } from "react";
import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import dynamic from "next/dynamic";
import HeroSection from "../components/HeroSection";
import PopularSection from "../components/PopularSection";
import { Metadata } from "next";

function PageLoader() {
    return (
        <div className="min-vh-100 d-flex align-items-center justify-content-center">
            <div className="spinner-border text-primary" />
        </div>
    );
}

function stripHtml(html: string): string {
    if (!html) return "";
    return html.replace(/<[^>]*>?/gm, "").trim();
}

import { buildPageMeta } from "@/lib/seo/generalMeta";

export async function generateMetadata(): Promise<Metadata> {
    const res = await API_ENDPOINTS.PAGE(1);

    if (!res.ok) {
        return {
            title: "Home",
            description: "Best assignment help service in India",
        };
    }

    const data = await res.json();
    return buildPageMeta(data, "/", "Home");
}

// Lazy load below-the-fold sections to reduce initial bundle
const PathSection = dynamic(() => import("../components/PathSection"), { ssr: true });
// const FeaturesSection = dynamic(() => import("../components/FeaturesSection"), { ssr: true });
const InfoSection = dynamic(() => import("../components/InfoSection"), { ssr: true });
const IgnouCtaSection = dynamic(() => import("../components/IgnouCtaSection"), { ssr: true });
const ProjectCardsSection = dynamic(() => import("../components/ProjectCardsSection"), { ssr: true });
const CustomReportSection = dynamic(() => import("../components/CustomReportSection"), { ssr: true });
const TestimonialsSection = dynamic(() => import("../components/TestimonialsSection"), { ssr: true });
const BlogsReviewsSection = dynamic(() => import("../components/BlogsReviewsSection"), { ssr: true });
const KeyBenefitsSection = dynamic(() => import("../components/KeyBenefitsSection"), { ssr: true });

// Loading skeletons
const SectionSkeleton = () => <div style={{ height: "400px", backgroundColor: "#f0f0f0", borderRadius: "8px", marginBottom: "2rem" }} />;

async function fetchHomeData() {
    const [
        heroSectionBannersRes,
        popularSectionOneRes,
        popularSectionTwoRes,
        pathSectionRes,
        pathCardsRes,
        infoSubSectionsRes,
        infoSectionRes,
        ignouCtaSectionRes,
        projectCardsSectionRes,
        projectCardsRes,
        customReportSectionRes,
        blogReviewsSectionRes,
        blogsRes,
        keyBenefitsRes,
        companyRes,
        testimonialsSectionRes,
        testimonialsRes,
    ] = await Promise.all([
        API_ENDPOINTS.BANNERS(),
        API_ENDPOINTS.SECTION(51),
        API_ENDPOINTS.SECTION(1),
        API_ENDPOINTS.SECTION(12),
        API_ENDPOINTS.ASSIGNMENTS.CARDS(),
        API_ENDPOINTS.SUBSECTIONS(1, 13),
        API_ENDPOINTS.SECTION(17),
        API_ENDPOINTS.SUBSECTIONS(1, 22),
        API_ENDPOINTS.SECTION(26),
        API_ENDPOINTS.PROJECTS.CARDS(),
        API_ENDPOINTS.SECTION(52),
        API_ENDPOINTS.SECTION(56),
        API_ENDPOINTS.BLOGS.LIST(),
        API_ENDPOINTS.SUBSECTIONS(1, 57),
        API_ENDPOINTS.COMPANY_CONTACT(),
        API_ENDPOINTS.SECTION(55),
        API_ENDPOINTS.TESTIMONIALS(),
    ]);

    if (!heroSectionBannersRes.ok || !popularSectionOneRes.ok || !popularSectionTwoRes.ok || !pathSectionRes.ok || !pathCardsRes.ok || !infoSubSectionsRes.ok || !infoSectionRes.ok || !ignouCtaSectionRes.ok || !projectCardsSectionRes.ok || !projectCardsRes.ok || !customReportSectionRes.ok || !blogReviewsSectionRes.ok || !blogsRes.ok || !keyBenefitsRes.ok || !companyRes.ok || !testimonialsSectionRes.ok || !testimonialsRes.ok) {
        throw new Error("Failed to load About page data");
    }

    const heroSectionBannersDataRaw = await heroSectionBannersRes.json();
    const heroSectionBannersData = heroSectionBannersDataRaw.map(
        (slide: any) => ({
            id: slide.id,
            title: slide.title,
            subtitle: slide.subtitle,
            text: slide.text,
            images: slide.images,
            ctaPrimary: slide.ctaPrimary,
            ctaSecondary: slide.ctaSecondary,
        })
    );
    const popularSectionOneData = await popularSectionOneRes.json();
    const popularSectionTwoData = await popularSectionTwoRes.json();
    const pathCardsSectionData = await pathSectionRes.json();

    const pathCardsListRaw = await pathCardsRes.json();
    const pathCardsList = Array.isArray(pathCardsListRaw.cards) ? pathCardsListRaw.cards : [];

    const infoSubSectionsData = await infoSubSectionsRes.json();
    const infoSectionData = await infoSectionRes.json();
    const ignouCtaSectionData = await ignouCtaSectionRes.json();
    const projectCardsSectionData = await projectCardsSectionRes.json();

    const projectCardsListRaw = await projectCardsRes.json();
    const projectCardsList = Array.isArray(projectCardsListRaw.cards) ? projectCardsListRaw.cards : [];

    const customReportSectionData = await customReportSectionRes.json();
    const blogReviewsSectionData = await blogReviewsSectionRes.json();
    const blogsList = await blogsRes.json();
    const keyBenefitsApi = await keyBenefitsRes.json();
    const companyApi = await companyRes.json();
    const testimonialsSectionData = await testimonialsSectionRes.json();
    const testimonialsList = await testimonialsRes.json();

    return {
        heroSectionBannersData,
        popularSectionData: {
            headlineOne: popularSectionOneData.section_headline,
            descriptionOne: popularSectionOneData.description,
            headlineTwo: popularSectionTwoData.section_headline,
            titleTwo: popularSectionTwoData.section_title,
            descriptionTwo: popularSectionTwoData.description,
            items: [
                { title: "Solved Assignments", img: "/img/project-1 (1).png", to: "/assignments/solved" },
                { title: "Project And Synopsis", img: "/img/project-1 (2).png", to: "/projects" },
                // { title: "Handwritten Scan Assignment", img: "/img/project-1 (3).png", to: "/assignments/hard-copy" },
                { title: "Hardcopy Assignment", img: "/img/project-1 (4).png", to: "/assignments/hard-copy" },
                { title: "Assignment Cover Page", img: "/img/project-1 (5).png", to: "/free-download" },
                { title: "Guess Paper", img: "/img/project-1 (6).png", to: "/guess-papers" },
                { title: "Free Download", img: "/img/project-1 (7).png", to: "/free-download" },
                { title: "FAQs", img: "/img/project-1 (8).png", to: "/faqs" },
            ],
        },
        pathCardsData: {
            headline: pathCardsSectionData.section_headline,
            description: pathCardsSectionData.description,
            button_name: pathCardsSectionData.button_name,
            pathCards: pathCardsList,
        },
        infoData: {
            parent: {
                title: infoSubSectionsData.parent_section_title,
                headline: infoSubSectionsData.parent_section_headline,
                description: infoSubSectionsData.parent_section_description,
                image: infoSubSectionsData.parent_section_image,
            },
            subsections: infoSubSectionsData.subsections || [],

            headlineTwo: infoSectionData.section_headline,
            descriptionTwo: infoSectionData.description,
            imageTwo: infoSectionData.section_image,
            buttonTwo: infoSectionData.button_name,
        },
        ignouCtaData: {
            subsections: ignouCtaSectionData.subsections || [],
        },
        projectCardsData: {
            headline: projectCardsSectionData.section_headline,
            description: projectCardsSectionData.description,
            button_name: projectCardsSectionData.button_name,
            projectCards: projectCardsList,
        },
        customReportData: {
            headline: customReportSectionData.section_headline,
            description: customReportSectionData.description,
            company: companyApi,
        },
        blogsData: {
            headline: blogReviewsSectionData.section_headline,
            description: blogReviewsSectionData.description,
            blogs: blogsList || [],
        },
        keyBenefitsData: {
            parent: {
                headline: keyBenefitsApi.parent_section_headline,
                description: keyBenefitsApi.parent_section_description,
            },
            subsections: keyBenefitsApi.subsections || [],
            company: companyApi,
        },
        testimonialsData: {
            headline: testimonialsSectionData.section_headline,
            description: testimonialsSectionData.description,
            testimonials: testimonialsList || [],
        },
    };
}

async function HomeContent() {
    const {
        heroSectionBannersData,
        popularSectionData,
        pathCardsData,
        infoData,
        ignouCtaData,
        projectCardsData,
        customReportData,
        blogsData,
        keyBenefitsData,
        testimonialsData,
    } = await fetchHomeData();

    return (
        <>
            <main>
                {/* Hero Section visible only on Home - loaded immediately */}
                <HeroSection data={heroSectionBannersData} />
                <PopularSection data={popularSectionData} />

                {/* Below-the-fold sections lazy loaded with suspense boundaries */}
                <Suspense fallback={<SectionSkeleton />}>
                    <PathSection data={pathCardsData} />
                </Suspense>

                {/* <Suspense fallback={<SectionSkeleton />}>
                    <FeaturesSection />
                </Suspense> */}

                <Suspense fallback={<SectionSkeleton />}>
                    <InfoSection data={infoData} />
                </Suspense>

                <Suspense fallback={<SectionSkeleton />}>
                    <IgnouCtaSection data={ignouCtaData} />
                </Suspense>

                <Suspense fallback={<SectionSkeleton />}>
                    <ProjectCardsSection data={projectCardsData} />
                </Suspense>

                <Suspense fallback={<SectionSkeleton />}>
                    <CustomReportSection data={customReportData} />
                </Suspense>

                <Suspense fallback={<SectionSkeleton />}>
                    <TestimonialsSection data={testimonialsData} />
                </Suspense>

                <Suspense fallback={<SectionSkeleton />}>
                    <BlogsReviewsSection data={blogsData} />
                </Suspense>

                <Suspense fallback={<SectionSkeleton />}>
                    <KeyBenefitsSection data={keyBenefitsData} />
                </Suspense>
            </main>
        </>
    );
}

export default function Home() {
    return (
        <Suspense
            fallback={<PageLoader />}
        >
            <HomeContent />
        </Suspense>
    );
}

