import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import ProjectsClient from "./ProjectsClient";
import { buildProjectListMeta } from "@/lib/seo/projectMeta";

export const dynamic = "force-dynamic";

async function fetchProjectsData() {
    try {
        const [pageRes, catRes] = await Promise.all([
            API_ENDPOINTS.PAGE(10),
            API_ENDPOINTS.PROJECTS.CATEGORIES_COURSES(),
        ]);

        if (!pageRes.ok || !catRes.ok) {
            throw new Error("Failed to fetch project data");
        }

        const [pageData, categoriesData] = await Promise.all([
            pageRes.json(),
            catRes.json(),
        ]);

        return { pageData, categoriesData };
    } catch (error) {
        console.error("Error fetching projects data:", error);
        return { pageData: null, categoriesData: null };
    }
}

export async function generateMetadata(): Promise<Metadata> {
    const { pageData } = await fetchProjectsData();
    if (!pageData) return { title: "Projects" };
    return buildProjectListMeta(pageData);
}

export default async function ProjectsPage() {
    const { pageData, categoriesData } = await fetchProjectsData();

    if (!pageData || !categoriesData) {
        return <div className="text-center py-5">Failed to load projects.</div>;
    }

    return <ProjectsClient pageData={pageData} categoriesData={categoriesData} />;
}
