"use client";

import React, { useEffect, useState } from "react";
import { AUTH, getToken } from "@/lib/apiEndpoints";
import InnerBanner from "../../components/InnerBanner";
import "../auth.css"; // Shared styles
import { Mail, Lock, Eye, EyeOff, User } from "lucide-react";
import { useRouter } from "next/navigation";

function PageLoader() {
    return (
        <div className="min-vh-100 d-flex align-items-center justify-content-center">
            <div className="spinner-border text-primary" />
        </div>
    );
}

type ErrorType = string | Record<string, string> | null;

export default function SignUpPage() {
    const router = useRouter();
    const auth = AUTH();

    const [checking, setChecking] = useState(true);
    useEffect(() => {
        const token = getToken();

        if (token) {
            router.replace("/dashboard");
        } else {
            setChecking(false);
        }
    }, [router]);

    const [formData, setFormData] = useState({
        first_name: "",
        last_name: "",
        email: "",
        password: "",
        confirmPassword: ""
    });
    const [error, setError] = useState<ErrorType>(null);
    const [loading, setLoading] = useState(false);
    const [showPassword, setShowPassword] = useState(false);
    const [showConfirmPassword, setShowConfirmPassword] = useState(false);
    const [acceptTerms, setAcceptTerms] = useState(false);

    function validate() {
        if (!formData.first_name) return "Please enter your First Name.";
        if (!formData.email || !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email)) return "Please enter a valid E-Mail.";
        if (!formData.password) return "Password is required.";
        if (!formData.password || formData.password.length < 6) return "Password must be at least 6 characters.";
        if (formData.password !== formData.confirmPassword) return "Passwords do not match.";
        if (!acceptTerms) return "Please accept terms & conditions.";
        return null;
    }

    async function handleSubmit(e: React.FormEvent) {
        e.preventDefault();
        setError(null);

        const validationError = validate();
        if (validationError) return setError({ general: validationError });

        if (!("SIGNUP" in auth) || !auth.SIGNUP) {
            return router.replace("/dashboard");
        }

        setLoading(true);

        try {
            const res = await auth.SIGNUP({
                first_name: formData.first_name,
                last_name: formData.last_name,
                email: formData.email,
                password: formData.password,
                confirm_password: formData.confirmPassword,
                terms: acceptTerms,
            });

            // ⭐️ treat token as success
            if (res?.access_token) {
                router.push("/dashboard");
                return;
            }

            if (res.errors) {
                setError(res.errors || {});
            } else {
                setError({ general: res.message || "Registration failed" });
            }
        } catch (err) {
            setError("Something went wrong. Please try again.");
        } finally {
            setLoading(false);
        }
    }

    function handleChange(field: string, value: string) {
        setFormData(prev => ({ ...prev, [field]: value }));
    }

    // 👇 render guard — SAFE, no hook order change
    if (checking) {
        return <PageLoader />;
    }

    return (
        <>
            <InnerBanner title="Sign Up" breadcrumb="Sign Up" />
            <div className="auth-page-wrapper">
                <div className="auth-card">
                    {/* Header */}
                    <div className="auth-card-header">
                        <h2 className="auth-title">Registration</h2>
                    </div>

                    <form onSubmit={handleSubmit} autoComplete="off">
                        {/* {error && (
                            <div className="auth-alert" style={{ marginBottom: '20px', color: 'red' }}>
                                <span>{error}</span>
                            </div>
                        )} */}
                        {error && (
                            <div className="auth-alert" style={{ marginBottom: '20px', color: 'red' }}>
                                {typeof error === "string" ? error : error.general ?? Object.values(error)[0]}
                            </div>
                        )}

                        {/* First Name */}
                        <div className="auth-form-group">
                            <div className="auth-input-wrapper">
                                <User className="auth-input-icon" size={20} />
                                <input
                                    className="auth-input"
                                    type="text"
                                    placeholder="Enter your First Name"
                                    value={formData.first_name}
                                    autoComplete="off"
                                    onChange={(e) => handleChange('first_name', e.target.value)}
                                />
                            </div>
                        </div>

                        {/* Last Name */}
                        <div className="auth-form-group">
                            <div className="auth-input-wrapper">
                                <User className="auth-input-icon" size={20} />
                                <input
                                    className="auth-input"
                                    type="text"
                                    placeholder="Enter your Last Name"
                                    value={formData.last_name}
                                    autoComplete="off"
                                    onChange={(e) => handleChange('last_name', e.target.value)}
                                />
                            </div>
                        </div>

                        {/* E-Mail */}
                        <div className="auth-form-group">
                            <div className="auth-input-wrapper">
                                <Mail className="auth-input-icon" size={20} />
                                <input
                                    className="auth-input"
                                    type="email"
                                    placeholder="Enter your E-Mail"
                                    value={formData.email}
                                    autoComplete="off"
                                    onChange={(e) => handleChange('email', e.target.value)}
                                />
                            </div>
                        </div>

                        {/* Password */}
                        <div className="auth-form-group">
                            <div className="auth-input-wrapper">
                                <Lock className="auth-input-icon" size={20} />
                                <input
                                    className="auth-input"
                                    type={showPassword ? "text" : "password"}
                                    placeholder="Create a Password"
                                    value={formData.password}
                                    autoComplete="new-password"
                                    onChange={(e) => handleChange('password', e.target.value)}
                                />
                                <button
                                    type="button"
                                    className="auth-password-toggle"
                                    onClick={() => setShowPassword(!showPassword)}
                                >
                                    {showPassword ? <EyeOff size={18} /> : <Eye size={18} />}
                                </button>
                                {/* Image doesn't show eye icon on first password field strictly, but good UX to have it. I'll adhere to image if strict, but toggles are standard. Image shows eye on confirm only? I'll put on both for UX. */}
                            </div>
                        </div>

                        {/* Confirm Password */}
                        <div className="auth-form-group">
                            <div className="auth-input-wrapper">
                                <Lock className="auth-input-icon" size={20} />
                                <input
                                    className="auth-input"
                                    type={showConfirmPassword ? "text" : "password"}
                                    placeholder="Confirm a Password"
                                    value={formData.confirmPassword}
                                    autoComplete="new-password"
                                    onChange={(e) => handleChange('confirmPassword', e.target.value)}
                                />
                                <button
                                    type="button"
                                    className="auth-password-toggle"
                                    onClick={() => setShowConfirmPassword(!showConfirmPassword)}
                                >
                                    {showConfirmPassword ? <EyeOff size={18} /> : <Eye size={18} />}
                                </button>
                            </div>
                        </div>

                        {/* Terms */}
                        <div className="auth-options">
                            <label className="auth-checkbox-label">
                                <input
                                    type="checkbox"
                                    className="auth-checkbox"
                                    checked={acceptTerms}
                                    onChange={(e) => setAcceptTerms(e.target.checked)}
                                />
                                <span>I accept all terms & conditions</span>
                            </label>
                        </div>

                        {/* Submit */}
                        <button
                            type="submit"
                            className="btn-auth"
                            disabled={loading}
                        >
                            {loading ? "Registering..." : "Register Now"}
                        </button>

                        {/* Footer Link */}
                        <div className="auth-footer-link">
                            Already have an account? <a href="/login" className="auth-link">Login now</a>
                        </div>
                    </form>
                </div>
            </div>
        </>
    );
}