"use client";

import React, { useState } from "react";
import { motion, Variants } from "framer-motion";
import { usePathname } from "next/navigation";
import "./Footer.css";
import { API_ENDPOINTS } from "@/lib/apiEndpoints";

interface FooterLink {
    label: string;
    href: string;
}

interface ContactItem {
    icon: string;
    text: string;
}

interface LinkColumn {
    title: string;
    links?: FooterLink[];
    contacts?: ContactItem[];
}

/* ANIMATION VARIANTS */
const sectionVariant: Variants = {
    hidden: { opacity: 0, y: 50 },
    visible: { opacity: 1, y: 0, transition: { duration: 0.8 } },
};

const columnVariant: Variants = {
    hidden: { opacity: 0, y: 30 },
    visible: (i: number) => ({
        opacity: 1,
        y: 0,
        transition: { delay: i * 0.2, duration: 0.6 },
    }),
};

function stripHtml(html: string): string {
    if (typeof window !== "undefined") {
        const div = document.createElement("div");
        div.innerHTML = html;
        return div.textContent || div.innerText || "";
    }
    return html.replace(/<[^>]*>?/gm, "").trim();
}

interface FooterProps {
    data: {
        company: any;
    } | null;
}

const Footer: React.FC<FooterProps> = ({ data }) => {
    const pathname = usePathname();
    const isLoginPage = pathname === "/login";
    const isRegisterPage = pathname === "/register";
    const isContactPage = pathname === "/contact";
    const isDashboardPage = pathname === "/dashboard";

    /* FORM STATES */
    const [formData, setFormData] = useState({
        first_name: "",
        last_name: "",
        email: "",
        phone: "",
        message: "",
    });

    const [loading, setLoading] = useState(false);
    const [error, setError] = useState("");
    const [success, setSuccess] = useState("");

    /* Supports input + textarea + select */
    const handleChange = (
        e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>
    ) => {
        const { name, value } = e.target;

        let updatedValue = value;

        // Allow only digits for phone field
        if (name === "phone") {
            updatedValue = value
                .replace(/\D/g, "")
                .slice(0, 15);
        }

        setFormData({
            ...formData,
            [name]: updatedValue,
        });
    };

    /* SUBMIT FORM */
    const handleSubmit = async () => {
        setError("");
        setSuccess("");

        if (!formData.first_name || !formData.email || !formData.phone) {
            setError("Please fill all required fields.");
            return;
        }

        try {
            setLoading(true);

            const form = new FormData();
            form.append("first_name", formData.first_name);
            form.append("last_name", formData.last_name);
            form.append("email", formData.email);
            form.append("phone", formData.phone);
            form.append("message", formData.message);

            const res = await API_ENDPOINTS.CONTACT_ENQUIRY_SAVE(form);

            const data = await res.json();

            if (!res.ok) {
                throw new Error(data.message || "Something went wrong");
            }

            setSuccess("Message sent successfully!");
            setFormData({
                first_name: "",
                last_name: "",
                email: "",
                phone: "",
                message: "",
            });

        } catch (err: any) {
            setError(err.message || "Failed to send message.");
        } finally {
            setLoading(false);
        }
    };

    /* FOOTER COLUMNS */
    const columns: LinkColumn[] = [
        {
            title: "Useful links",
            links: [
                { label: "Home", href: "/" },
                { label: "About Us", href: "/about" },
                { label: "Blogs", href: "/blog" },
                { label: "Contact Us", href: "/contact" },
                { label: "FAQs", href: "/faqs" },
            ],
        },
        {
            title: "Our Services",
            links: [
                { label: "IGNOU Solved Assignments", href: "/assignments/solved" },
                // { label: "IGNOU Handwritten Assignments", href: "/assignments/handwritten" },
                { label: "IGNOU Guess Papers", href: "/guess-papers" },
                { label: "IGNOU Handwritten Assignments", href: "/assignments/hard-copy" },
                { label: "IGNOU Projects", href: "/projects" },
            ],
        },
        {
            title: "Help & Support",
            links: [
                { label: "Terms & Conditions", href: "/" },
                { label: "Privacy Policy", href: "/" },
                { label: "Refund Policy", href: "/" },
            ],
        },
        {
            title: "Contact Us",
            contacts: [
                { icon: "fa-location-dot", text: stripHtml(data?.company.location) },
                { icon: "fa-envelope", text: data?.company.email },
                { icon: "fa-phone", text: data?.company.phone },
            ],
        },
    ];

    return (
        <footer className="exc-footer">

            {/* CTA + CONTACT FORM - HIDE ON CONTACT PAGE */}
            {!isLoginPage && !isRegisterPage && !isContactPage && !isDashboardPage && (
                <motion.div
                    className="exc-footer-cta text-white overflow-hidden"

                    variants={sectionVariant}
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                >
                    <div className="container pt-5">
                        <div className="row align-items-center g-4">

                            {/* LEFT CONTENT + FORM */}
                            <div className="col-lg-6 col-md-7">

                                {/* <h3 className="exc-footer-cta-heading mb-2">Stay ahead with the latest IGNOU</h3>
                                <p className="exc-footer-cta-text mb-3">
                                    Get updates, exclusive offers and special deals by joining our WhatsApp & Telegram.
                                </p> */}

                                {/* CONTACT FORM */}
                                <div className="contact-form-wrapper py-5 px-4 text-dark  mb-5">

                                    <h3 className="form-title mb-2 fw-bold">
                                        Get in Touch
                                    </h3>
                                    <p className="text-muted mb-3 ">Have questions? Send us a message!</p>

                                    {error && (
                                        <div className="alert alert-danger py-2">
                                            <i className="fa-solid fa-circle-exclamation me-2"></i>{error}
                                        </div>
                                    )}

                                    {success && (
                                        <div className="alert alert-success py-2">
                                            <i className="fa-solid fa-circle-check me-2"></i>{success}
                                        </div>
                                    )}

                                    <div className="row">

                                        {/* First Name */}
                                        <div className="col-lg-6 mb-3">
                                            <input
                                                type="text"
                                                name="first_name"
                                                className="form-control"
                                                value={formData.first_name}
                                                onChange={handleChange}
                                                placeholder="Enter your First Name"
                                            />
                                        </div>

                                        {/* Last Name */}
                                        <div className="col-lg-6 mb-3">
                                            <input
                                                type="text"
                                                name="last_name"
                                                className="form-control"
                                                value={formData.last_name}
                                                onChange={handleChange}
                                                placeholder="Enter your Last Name"
                                            />
                                        </div>

                                        {/* Email */}
                                        <div className="col-lg-6 mb-3">
                                            <input
                                                type="email"
                                                name="email"
                                                className="form-control"
                                                value={formData.email}
                                                onChange={handleChange}
                                                placeholder="Enter your E-Mail"
                                            />
                                        </div>

                                        {/* Phone */}
                                        <div className="col-lg-6 mb-3">
                                            <input
                                                type="text"
                                                name="phone"
                                                className="form-control"
                                                value={formData.phone}
                                                onChange={handleChange}
                                                placeholder="Enter your Phone"
                                            />
                                        </div>

                                        {/* Subject Type */}
                                        {/* <div className="col-lg-6 mb-3">
                                            <select
                                                name="subjectType"
                                                className="form-control"
                                                value={formData.subjectType}
                                                onChange={handleChange}
                                            >
                                                <option value="">Select Subject</option>
                                                <option value="Bachelor">Bachelor</option>
                                                <option value="Master">Master</option>
                                                <option value="Diploma">Diploma</option>
                                                <option value="PG Diploma">PG Diploma</option>
                                                <option value="Certificate">Certificate</option>
                                            </select>
                                        </div> */}

                                        {/* Message */}
                                        <div className="col-lg-12 mb-3">
                                            <textarea
                                                name="message"
                                                className="form-control"
                                                rows={3}
                                                value={formData.message}
                                                onChange={handleChange}
                                                placeholder="Write your Message..."
                                            ></textarea>
                                        </div>

                                        {/* Submit */}
                                        <div className="col-lg-12 m-auto">
                                            <button
                                                className="btn btn-primary w-50 py-2"
                                                onClick={handleSubmit}
                                                disabled={loading}
                                            >
                                                {loading ? "Sending..." : "Send Message"}
                                            </button>
                                        </div>

                                    </div>
                                </div>
                            </div>

                            {/* RIGHT SIDE IMAGE */}
                            <div className="col-lg-6 col-md-5 text-md-end">
                                <div className="exc-footer-cta-badge mb-3">
                                    <span className="exc-badge-label">Top Rating Platform</span>
                                    <span className="exc-badge-stars">★★★★★</span>
                                </div>
                                <img src="/img/footer-1.png" className="animation-shape" width="300" alt="Footer" />
                                <img src="/img/thumb.png" className="footer-img-top" width="300" alt="Footer" />
                            </div>

                        </div>
                    </div>
                </motion.div>
            )}

            {/* MIDDLE COLUMNS */}
            <motion.div
                className="exc-footer-main bg-white"
                variants={sectionVariant}
                initial="hidden"
                whileInView="visible"
                viewport={{ once: true }}
            >
                <div className="container py-5">
                    <div className="row g-4 g-md-5">
                        {columns.map((col, i) => (
                            <motion.div
                                key={i}
                                className="col-md-6 col-lg-3 col-sm-12"
                                custom={i}
                                variants={columnVariant}
                                initial="hidden"
                                whileInView="visible"
                                viewport={{ once: true }}
                            >
                                <h6 className="exc-footer-title">{col.title}</h6>

                                {col.links && (
                                    <ul className="list-unstyled exc-footer-list">
                                        {col.links.map((link, j) => (
                                            <li key={j}>
                                                <a href={link.href} className="exc-footer-link">{link.label}</a>
                                            </li>
                                        ))}
                                    </ul>
                                )}

                                {col.contacts && (
                                    <ul className="list-unstyled exc-footer-list mb-0">
                                        {col.contacts.map((c, j) => (
                                            <li key={j} className="exc-footer-text">
                                                <i className={`fa-solid ${c.icon} me-2`}></i> {c.text}
                                            </li>
                                        ))}
                                    </ul>
                                )}
                            </motion.div>
                        ))}
                    </div>
                </div>
            </motion.div>

            {/* BOTTOM BAR */}
            <motion.div
                className="exc-footer-bottom bg-white border-top"
                variants={sectionVariant}
                initial="hidden"
                whileInView="visible"
                viewport={{ once: true }}
            >
                <div className="container py-4">
                    <div className="row align-items-center g-3">

                        <div className="col-md-6 d-lg-flex d-md-block d-block align-items-center gap-2">
                            <img src="/img/footer-img.png" width="80" alt="Footer logo" />
                            <div className="footer-info-logo">
                                <div className="d-flex gap-2">
                                    Follow Us On:
                                    {["facebook-f", "instagram", "youtube", "telegram", "whatsapp"].map((icon) => (
                                        <a key={icon} href="#" className="exc-footer-social-icon">
                                            <i className={`fa-brands fa-${icon}`}></i>
                                        </a>
                                    ))}
                                </div>
                                <div className="exc-footer-copy mt-2">
                                    © {new Date().getFullYear()} Matebiz.com. All Rights Reserved.
                                </div>
                            </div>
                        </div>

                        <div className="col-md-6 text-md-end ">
                            {/* <button className="exc-footer-btn exc-footer-btn-outline ">Contact Us</button> */}
                            <button className="exc-footer-btn exc-footer-btn-outline mx-3">Scroll to Top</button>
                            <button className="exc-footer-btn exc-footer-btn-primary">Help Desk</button>
                        </div>

                    </div>
                </div>
            </motion.div>

        </footer>
    );
};

export default Footer;
