"use client";

import React from "react";
import Image from "next/image";
import Link from "next/link";
import { motion } from "framer-motion";
import "./InfoSection.css";

interface SubSection {
    title: string;
    icon?: string;
}

interface InfoData {
    parent: {
        title: string;
        headline: string;
        description: string;
        image?: string;
        video?: string;
    };
    subsections: SubSection[];
    headlineTwo: string;
    descriptionTwo: string;
    imageTwo?: string;
    buttonTwo: string;
}

interface InfoSectionProps {
    data: InfoData | null;
}

// Animation Variants
const fadeUp = {
    hidden: { opacity: 0, y: 40 },
    show: { opacity: 1, y: 0, transition: { duration: 0.6 } },
};

const fadeLeft = {
    hidden: { opacity: 0, x: -50 },
    show: { opacity: 1, x: 0, transition: { duration: 0.6 } },
};

const fadeRight = {
    hidden: { opacity: 0, x: 50 },
    show: { opacity: 1, x: 0, transition: { duration: 0.6 } },
};

const staggerParent = {
    hidden: {},
    show: { transition: { staggerChildren: 0.2 } },
};

export default function InfoSection({ data }: InfoSectionProps) {
    if (!data) return null;

    const { parent, subsections, headlineTwo, descriptionTwo, imageTwo, buttonTwo } = data;

    return (
        <>
            {/* ========================== INFO SECTION ========================== */}
            <section className="pt-5 info-section">
                <div className="container">
                    <motion.div
                        className="row align-items-center"
                        initial="hidden"
                        whileInView="show"
                        viewport={{ once: true }}
                    >
                        {/* Left Title */}
                        <motion.div className="col-lg-6 mb-4 mb-lg-0" variants={fadeLeft}>
                            <h3 className="info-title fw-bold mb-3">{parent.headline}</h3>
                            <div dangerouslySetInnerHTML={{ __html: parent.description }} />
                        </motion.div>

                        {/* Info Items */}
                        <motion.div className="col-lg-12 mt-4" variants={staggerParent}>
                            <div className="row">
                                {subsections.map((item, index) => (
                                    <motion.div key={index} className="col-lg-4 col-sm-6 mb-4" variants={fadeUp}>
                                        <div className="info-flex d-flex align-items-center gap-3">
                                            {item.icon && (
                                                <motion.div
                                                    className="info-icon"
                                                    initial={{ scale: 0.8, opacity: 0 }}
                                                    animate={{ scale: 1, opacity: 1 }}
                                                    transition={{ duration: 0.4 }}
                                                    viewport={{ once: true }}
                                                >
                                                    <Image src={item.icon} alt={item.title} width={40} height={40} unoptimized />
                                                </motion.div>
                                            )}
                                            <h6 className="m-0">{item.title}</h6>
                                        </div>
                                    </motion.div>
                                ))}
                            </div>
                        </motion.div>
                    </motion.div>
                </div>
            </section>

            {/* ========================== INFO ABOUT SECTION ========================== */}
            <section className="py-5 info-about-section">
                <div className="container">
                    <motion.div
                        className="row align-items-center"
                        initial="hidden"
                        whileInView="show"
                        viewport={{ once: true }}
                    >
                        {/* Left Content */}
                        <motion.div className="col-lg-6 mb-4 mb-lg-0" variants={fadeLeft}>
                            <h3 className="info-title fw-bold mb-3">{headlineTwo}</h3>
                            <div className="text-muted" dangerouslySetInnerHTML={{ __html: descriptionTwo }} />

                            <div className="info-btn mt-4">
                                <Link href="#" className="btn btn-primary rounded-pill px-4">
                                    {buttonTwo} &nbsp;
                                    <i className="fa-solid fa-angle-right"></i>
                                </Link>
                            </div>
                        </motion.div>

                        {/* Right Image */}
                        <motion.div className="col-lg-6" variants={fadeRight}>
                            {imageTwo && (
                                <motion.div
                                    className="info-ab-img text-center"
                                    initial={{ opacity: 0, scale: 0.9 }}
                                    whileInView={{ opacity: 1, scale: 1 }}
                                    transition={{ duration: 0.7 }}
                                    viewport={{ once: true }}
                                >
                                    <Image
                                        src={imageTwo}
                                        alt={headlineTwo}
                                        width={500}
                                        height={450}
                                        className="img-fluid rounded-3"
                                        style={{ width: "100%", height: "auto" }}
                                        unoptimized
                                    />
                                </motion.div>
                            )}
                        </motion.div>
                    </motion.div>
                </div>
            </section>
        </>
    );
}
