"use client";

import React from "react";
import { motion, Variants } from "framer-motion";
import "./KeyBenefitsSection.css";

const fadeUp: Variants = {
    hidden: { opacity: 0, y: 30 },
    show: {
        opacity: 1,
        y: 0,
        transition: { duration: 0.6, ease: "easeOut" },
    },
};

const cardAnim: Variants = {
    hidden: { opacity: 0, y: 20, scale: 0.96 },
    show: {
        opacity: 1,
        y: 0,
        scale: 1,
        transition: { duration: 0.4, ease: "easeOut" },
    },
};

const staggerCards: Variants = {
    hidden: {},
    show: {
        transition: { staggerChildren: 0.12 },
    },
};

interface BenefitItem {
    title: string;
    icon: string;
}

interface CompanyContact {
    telegram_url?: string;
    whatsapp_url?: string;
}

interface KeyBenefitsData {
    parent: {
        headline: string;
        description: string;
    };
    subsections: BenefitItem[];
    company: CompanyContact | null;
}

interface KeyBenefitsSectionProps {
    data: KeyBenefitsData | null;
}

function stripHtml(html: string): string {
    if (!html) return "";
    return html.replace(/<[^>]*>?/gm, "").trim();
}

export default function KeyBenefitsSection({ data }: KeyBenefitsSectionProps) {
    if (!data || !data.parent || !data.subsections?.length) {
        return null;
    }

    const { parent, subsections, company } = data;

    return (
        <section className="kbs-wrapper py-5">
            <motion.div
                className="container text-center"
                initial="hidden"
                whileInView="show"
                viewport={{ once: true, amount: 0.2 }}
            >
                {/* Heading */}
                <motion.div variants={fadeUp} className="mb-4">
                    <h2 className="brs-heading mb-2">
                        {parent.headline}
                    </h2>

                    <div
                        className="kbs-subtext mb-0"
                        dangerouslySetInnerHTML={{ __html: stripHtml(parent.description) }}
                    />
                </motion.div>

                {/* Benefit Cards */}
                <motion.div
                    className="row g-3 g-md-4 justify-content-center mb-5"
                    variants={staggerCards}
                >
                    {subsections.map((item, idx) => (
                        <motion.div
                            key={idx}
                            variants={cardAnim}
                            className="col-lg-3 col-md-6 col-sm-12 d-flex"
                        >
                            <div className="kbs-card flex-grow-1 rounded-4 p-3 p-md-4 d-flex flex-column align-items-center shadow">
                                <div className="kbs-card-icon mb-3 d-flex align-items-center justify-content-center">
                                    <img
                                        src={item.icon}
                                        alt={item.title}
                                        className="kbs-benefit-img"
                                        width={100}
                                        height={100}
                                        loading="lazy"
                                    />
                                </div>
                                <p className="kbs-card-title mb-0">
                                    {item.title}
                                </p>
                            </div>
                        </motion.div>
                    ))}
                </motion.div>

                {/* Subscribe Buttons */}
                {(company?.telegram_url || company?.whatsapp_url) && (
                    <motion.div
                        className="row g-4 justify-content-center align-items-center kbs-subscribe-row"
                        variants={fadeUp}
                    >
                        {company?.telegram_url && (
                            <div className="col-lg-6 col-md-12 col-sm-12 d-flex justify-content-center justify-content-lg-end">
                                <a
                                    href={company.telegram_url}
                                    target="_blank"
                                    rel="noopener noreferrer"
                                    className="kbs-sub-btn d-inline-flex align-items-center gap-3"
                                >
                                    <img
                                        src="/img/telegram.png"
                                        alt="Telegram"
                                        className="kbs-sub-img"
                                        width={32}
                                        height={32}
                                        loading="lazy"
                                    />
                                    <span className="kbs-sub-text">
                                        Subscribe to Our Telegram Channel
                                    </span>
                                </a>
                            </div>
                        )}

                        {company?.whatsapp_url && (
                            <div className="col-lg-6 col-md-12 col-sm-12 d-flex justify-content-center justify-content-lg-start">
                                <a
                                    href={company.whatsapp_url}
                                    target="_blank"
                                    rel="noopener noreferrer"
                                    className="kbs-sub-btn d-inline-flex align-items-center gap-3"
                                >
                                    <img
                                        src="/img/apple.png"
                                        alt="WhatsApp"
                                        className="kbs-sub-img"
                                        width={32}
                                        height={32}
                                        loading="lazy"
                                    />
                                    <span className="kbs-sub-text">
                                        Subscribe to Our WhatsApp Channel
                                    </span>
                                </a>
                            </div>
                        )}
                    </motion.div>
                )}
            </motion.div>
        </section>
    );
}
