"use client";

import React, { useEffect, useState } from "react";
import { AUTH, getToken } from "@/lib/apiEndpoints";
import Link from "next/link";
import styles from "./Navbar.module.css";
import { useCart } from "../context/CartContext";
import { useRouter } from "next/navigation";
import { Navbar as BootstrapNavbar, Nav, Offcanvas, Container } from 'react-bootstrap';

type NavbarProps = {
    data: {
        company: any;
    } | null;
}

export default function Navbar({ data }: NavbarProps) {
    const router = useRouter();
    const auth = AUTH();

    const { openCart, cartCount } = useCart();
    const [alertMessage, setAlertMessage] = useState<string>("");
    const [assignmentsOpen, setAssignmentsOpen] = useState(false);
    const [showOffcanvas, setShowOffcanvas] = useState(false);
    const [isLoggedIn, setIsLoggedIn] = useState(false);
    const [checking, setChecking] = useState(true);

    // Render the alert only on the client
    useEffect(() => {
        // setAlertMessage("🚨 Alert! 👉 IGNOU Assignment Submission Deadline: 30th September 2025 📚 ✍️");
        if (data?.company?.alert_message) {
            setAlertMessage(data.company.alert_message);
        }
    }, [data]);

    // 🔐 Check login + listen for auth changes
    useEffect(() => {
        const checkAuth = () => {
            try {
                const token = getToken();
                setIsLoggedIn(!!token);
            } catch {
                setIsLoggedIn(false);
            } finally {
                setChecking(false);
            }
        };

        checkAuth();

        // triggered after login/logout
        window.addEventListener("auth-changed", checkAuth);

        // sync if token changed in another tab
        window.addEventListener("storage", checkAuth);

        return () => {
            window.removeEventListener("auth-changed", checkAuth);
            window.removeEventListener("storage", checkAuth);
        };
    }, []);

    // Cleanup unused CSS preload links after load to avoid "preload but not used" warnings
    useEffect(() => {
        const id = setTimeout(() => {
            try {
                const preloads = Array.from(document.querySelectorAll('link[rel="preload"][as="style"]'));
                preloads.forEach((link) => {
                    const href = link.getAttribute("href");
                    if (!href) return;

                    // Consider it used if there's an actual stylesheet link with same href
                    const stylesheetLink = document.querySelector(`link[rel="stylesheet"][href="${href}"]`);
                    const applied = !!stylesheetLink || Array.from(document.styleSheets).some((s) => s.href === href);
                    if (!applied) {
                        link.remove();
                    }
                });
            } catch (e) {
                // ignore any DOM access errors
            }
        }, 3000);

        return () => clearTimeout(id);
    }, []);

    async function logout() {
        try {
            if (auth.LOGOUT) {
                await auth.LOGOUT();
            }

            sessionStorage.removeItem("buy_now_item");

            router.replace("/login");
        } catch (err) {
            console.error("Logout failed", err);
        }
    }

    return (
        <div className="sticky-lg-top shadow-lg">
            {/* 🔹 Top Alert Bar */}
            <div className="bg-black text-white py-1">
                <div className="container d-flex justify-content-between align-items-center">
                    {/* Marquee */}
                    <div className="marquee-container w-50 overflow-hidden">
                        {alertMessage && (
                            <div
                                className={styles.marquee}
                                onMouseEnter={(e) =>
                                    (e.currentTarget.style.animationPlayState = "paused")
                                }
                                onMouseLeave={(e) =>
                                    (e.currentTarget.style.animationPlayState = "running")
                                }
                            >
                                {alertMessage}
                            </div>
                        )}
                    </div>

                    {/* Sign In + Cart */}
                    <div className={`btn-sign-login d-flex gap-3 align-items-center ${styles.topBar}`}>
                        <div className="btn-sign-login">
                            {/* /login */}
                            <div className=" d-flex gap-2">
                                {!checking ? (
                                    isLoggedIn ? (
                                        <>
                                            <Link
                                                href="/dashboard"
                                                className={`btn border rounded-pill px-3 py-1 text-white ${styles.actionButton}`}
                                            >
                                                <i className="fa-solid fa-gauge-high me-2" />
                                                Dashboard
                                            </Link>
                                            <button
                                                onClick={logout}
                                                className={`btn border rounded-pill px-3 py-1 text-white ${styles.actionButton}`}
                                                type="button"
                                            >
                                                <i className="fa-solid fa-right-from-bracket me-2" />
                                                Logout
                                            </button>
                                        </>
                                    ) : (
                                        <Link
                                            href="/login"
                                            className={`btn-sign btn border rounded-pill px-3 py-1 text-white ${styles.actionButton}`}
                                        >
                                            <i className="fa-solid fa-user me-2" />
                                            Sign In
                                        </Link>
                                    )
                                ) : (
                                    <div style={{ padding: 20 }}></div>
                                )}
                            </div>
                        </div>

                        {/* /cart */}
                        <button
                            onClick={openCart}
                            className={`btn-cart btn border rounded-pill px-3 py-1 text-white ${styles.actionButton} position-relative`}
                        >
                            <i className="fa-solid fa-cart-shopping me-2"></i>
                            <span>
                                My Cart
                            </span>

                            {cartCount > 0 && (
                                <span className="cart-btn-numbrt">
                                    {cartCount}
                                    <span className="visually-hidden">Unread Messages</span>
                                </span>
                            )}
                        </button>
                    </div>
                </div>
            </div>

            {/* 🔹 Main Navigation Bar */}
            <BootstrapNavbar
                expand="xl"
                className={styles.mainNavbar}
                style={{ background: "linear-gradient(to right, #2f80ff, #5596ffff)" }}
            >
                <Container>
                    {/* Logo */}
                    <BootstrapNavbar.Brand
                        as={Link}
                        href="/"
                        className={`text-white fw-bold d-flex align-items-center ${styles.navbarBrand}`}
                    >
                        <img src="/img/logo.png" alt="Logo" width="150" height="auto" className="me-2" />
                    </BootstrapNavbar.Brand>

                    {/* Toggle */}
                    <BootstrapNavbar.Toggle
                        aria-controls="offcanvasNavbar"
                        className="bg-white"
                        onClick={() => setShowOffcanvas(true)}
                    />

                    {/* Offcanvas Menu */}
                    <BootstrapNavbar.Offcanvas
                        id="offcanvasNavbar"
                        aria-labelledby="offcanvasNavbarLabel"
                        placement="end"
                        show={showOffcanvas}
                        onHide={() => setShowOffcanvas(false)}
                    >
                        <Offcanvas.Header closeButton>
                            <Offcanvas.Title id="offcanvasNavbarLabel">
                                <BootstrapNavbar.Brand
                                    as={Link}
                                    href="/"
                                    className={`text-white fw-bold d-flex align-items-center ${styles.navbarBrand}`}
                                >
                                    <img src="/img/logo.png" alt="Logo" width="150" height="auto" className="me-2" />
                                </BootstrapNavbar.Brand>

                            </Offcanvas.Title>
                        </Offcanvas.Header>
                        <Offcanvas.Body>
                            <Nav className="justify-content-end flex-grow-1  ">
                                <Nav.Link
                                    as={Link}
                                    href="/"
                                    className={`${styles.responsiveNavLink}`}
                                    onClick={() => setShowOffcanvas(false)}
                                >
                                    Home
                                </Nav.Link>

                                {/* Custom Dropdown */}
                                <div
                                    className={`nav-item position-relative ${styles.navDropdown} ${assignmentsOpen ? styles.show : ""}`}
                                    onMouseEnter={() => {
                                        if (window.innerWidth >= 1200) setAssignmentsOpen(true);
                                    }}
                                    onMouseLeave={() => {
                                        if (window.innerWidth >= 1200) setAssignmentsOpen(false);
                                    }}
                                >
                                    <div className="d-flex align-items-center">
                                        {/* TEXT */}
                                        <a
                                            href="#"
                                            className={`nav-link ${styles.responsiveNavLink} ${styles.dropdownToggle}`}
                                            role="button"
                                            aria-expanded={assignmentsOpen}
                                            onClick={(e) => {
                                                // prevent text click navigation
                                                e.preventDefault();

                                                // mobile only toggle
                                                if (window.innerWidth < 1200) {
                                                    setAssignmentsOpen((s) => !s);
                                                }
                                            }}
                                        >
                                            IGNOU Assignments
                                        </a>

                                        {/* ARROW BUTTON (CLICK ONLY) */}
                                        <span
                                            className={`${styles.clk_btn} ${styles.responsiveNavLink}`}
                                            onClick={(e) => {
                                                e.preventDefault();
                                                setAssignmentsOpen((s) => !s);
                                            }}
                                        >
                                            <i
                                                className={`fa fa-sort-desc ${assignmentsOpen ? styles.rotate : ""
                                                    }`}
                                            ></i>
                                        </span>
                                    </div>

                                    {/* DROPDOWN MENU */}
                                    <ul className={`${styles.dropdownMenu}`}>
                                        <li>
                                            <Link
                                                href="/assignments/solved"
                                                className={styles.dropdownItem}
                                                onClick={() => {
                                                    setAssignmentsOpen(false);
                                                    setShowOffcanvas(false);
                                                }}
                                            >
                                                IGNOU Solved Assignments
                                            </Link>
                                        </li>
                                        <li>
                                            <Link
                                                href="/guess-papers"
                                                className={styles.dropdownItem}
                                                onClick={() => {
                                                    setAssignmentsOpen(false);
                                                    setShowOffcanvas(false);
                                                }}
                                            >
                                                IGNOU Guess Papers
                                            </Link>
                                        </li>
                                        <li>
                                            <Link
                                                href="/assignments/hard-copy"
                                                className={styles.dropdownItem}
                                                onClick={() => {
                                                    setAssignmentsOpen(false);
                                                    setShowOffcanvas(false);
                                                }}
                                            >
                                                IGNOU Handwritten Assignments
                                            </Link>
                                        </li>
                                    </ul>
                                </div>

                                <Nav.Link
                                    as={Link}
                                    href="/projects"
                                    className={`${styles.responsiveNavLink}`}
                                    onClick={() => setShowOffcanvas(false)}
                                >
                                    IGNOU Projects
                                </Nav.Link>
                                <Nav.Link
                                    as={Link}
                                    href="/blog"
                                    className={`${styles.responsiveNavLink}`}
                                    onClick={() => setShowOffcanvas(false)}
                                >
                                    Blogs
                                </Nav.Link>
                                <Nav.Link
                                    as={Link}
                                    href="/about"
                                    className={`${styles.responsiveNavLink}`}
                                    onClick={() => setShowOffcanvas(false)}
                                >
                                    About Us
                                </Nav.Link>
                                <Nav.Link
                                    as={Link}
                                    href="/contact"
                                    className={`${styles.responsiveNavLink}`}
                                    onClick={() => setShowOffcanvas(false)}
                                >
                                    Contact Us
                                </Nav.Link>
                            </Nav>

                            <div className={`btn-sign-login-mobile d-lg-flex d-lg-block d-block mt-4 gap-3 align-items-center ${styles.topBar}`}>
                                <div className="btn-sign-login-mobile">
                                    {/* /login */}
                                    <div className="d-lg-block ">
                                        {!checking ? (
                                            isLoggedIn ? (
                                                <>
                                                    <Link
                                                        href="/dashboard"
                                                        className={`btn btn-dark rounded-pill px-3 me-2 py-1 text-white ${styles.actionButton}`}
                                                    >
                                                        <i className="fa-solid fa-gauge-high me-2" />
                                                        Dashboard
                                                    </Link>
                                                    <button
                                                        onClick={logout}
                                                        className={`btn btn-dark rounded-pill px-3 py-1 text-white ${styles.actionButton}`}
                                                        type="button"
                                                    >
                                                        <i className="fa-solid fa-right-from-bracket me-2" />
                                                        Logout
                                                    </button>
                                                </>
                                            ) : (
                                                <Link
                                                    href="/login"
                                                    className={`btn btn-dark btn border rounded-pill px-3 py-1 text-white ${styles.actionButton}`}
                                                >
                                                    <i className="fa-solid fa-user me-2" />
                                                    Sign In
                                                </Link>
                                            )
                                        ) : (
                                            <div style={{ padding: 20 }}></div>
                                        )}
                                    </div>
                                </div>

                                {/* /cart */}

                            </div>
                        </Offcanvas.Body>
                    </BootstrapNavbar.Offcanvas>
                </Container>
            </BootstrapNavbar>
        </div>
    );
}
