"use client";

import React from "react";
import Link from "next/link";
import "./PathSection.css";
import { Swiper, SwiperSlide } from "swiper/react";
import { Autoplay } from "swiper/modules";
import { motion } from "framer-motion";
import "swiper/css";

interface PathCard {
    code: string;
    title: string;
    img: string;
    to: string;
}

interface PathSectionData {
    headline: string;
    description: string;
    button_name: string;
    pathCards: PathCard[];
}

interface PathSectionProps {
    data: PathSectionData | null;
}

// Animation Variants
const fadeUp = {
    hidden: { opacity: 0, y: 40 },
    show: { opacity: 1, y: 0, transition: { duration: 0.6, ease: "easeOut" as const } },
};

const slideIn = {
    hidden: { opacity: 0, x: 50 },
    show: { opacity: 1, x: 0, transition: { duration: 0.7, ease: "easeOut" as const } },
};

const cardAnim = {
    hidden: { opacity: 0, scale: 0.92, y: 30 },
    show: { opacity: 1, scale: 1, y: 0, transition: { duration: 0.45, ease: "easeOut" as const } },
};

// SSR-safe stripHtml
function stripHtml(html: string): string {
    if (typeof window === "undefined") {
        return html.replace(/<[^>]*>?/gm, "");
    }
    const div = document.createElement("div");
    div.innerHTML = html;
    return div.textContent || div.innerText || "";
}

export default function PathSection({ data }: PathSectionProps) {
    if (!data) return null;

    const { headline, description, button_name, pathCards } = data;

    const slidesForLoop =
        pathCards.length < 8 ? [...pathCards, ...pathCards] : pathCards;

    return (
        <section className="pb-5" style={{ background: "#f7faff" }}>
            <motion.div
                className="container"
                initial="hidden"
                whileInView="show"
                viewport={{ once: true, amount: 0.2 }}
            >
                {/* Heading Section */}
                <motion.div
                    className="d-flex flex-wrap align-items-center justify-content-between gap-3 mb-4"
                    variants={fadeUp}
                >
                    <div>
                        <h2 className="m-0 fw-bold mb-3">{headline}</h2>
                        <div
                            className="badge rounded-pill text-bg-primary-subtle text-primary px-3 py-2"
                            dangerouslySetInnerHTML={{ __html: stripHtml(description) }}
                        />
                    </div>

                    <Link
                        href="/assignments/solved"
                        className="btn btn-outline-dark rounded-pill px-4 d-flex align-items-center gap-2"
                    >
                        {button_name} <i className="fa-solid fa-chevron-right"></i>
                    </Link>
                </motion.div>

                {/* Slider Wrapper */}
                <motion.div variants={slideIn}>
                    <Swiper
                        modules={[Autoplay]}
                        autoplay={{ delay: 2500, disableOnInteraction: false }}
                        spaceBetween={16}
                        slidesPerView={4}
                        loop={slidesForLoop.length > 4}
                        breakpoints={{
                            0: { slidesPerView: 1.1, spaceBetween: 12 },
                            576: { slidesPerView: 2, spaceBetween: 16 },
                            992: { slidesPerView: 3, spaceBetween: 20 },
                            1200: { slidesPerView: 4, spaceBetween: 24 },
                        }}
                    >
                        {slidesForLoop.map((c, i) => (
                            <SwiperSlide key={i}>
                                <motion.div
                                    variants={cardAnim}
                                    initial="hidden"
                                    whileInView="show"
                                    viewport={{ once: true }}
                                    className="path-card h-100 rounded-4 shadow-sm p-3 bg-white"
                                >
                                    <div className="ratio ratio-16x9 mb-3 rounded-4 overflow-hidden">
                                        <img
                                            src={c.img}
                                            alt={c.title}
                                            className="w-100 h-100 object-fit-cover"
                                            width="300"
                                            height="200"
                                            loading="lazy"
                                        />
                                    </div>

                                    <div className="small text-muted">{c.code}</div>
                                    <div className="fw-semibold mb-3">{c.title}</div>

                                    <div className="d-flex justify-content-between align-items-center mt-auto">
                                        <Link
                                            href={c.to}
                                            className="path-btn btn btn-dark rounded-pill px-3 d-flex align-items-center gap-2"
                                        >
                                            More Details <i className="fa-solid fa-chevron-right"></i>
                                        </Link>
                                    </div>
                                </motion.div>
                            </SwiperSlide>
                        ))}
                    </Swiper>
                </motion.div>
            </motion.div>
        </section>
    );
}
