"use client";

import React from "react";
import { Swiper, SwiperSlide } from "swiper/react";
import { Autoplay } from "swiper/modules";
import { motion, type Variants } from "framer-motion";
import "swiper/css";
import "./TestimonialsSection.css";

const fadeUp: Variants = {
    hidden: { opacity: 0, y: 40 },
    show: {
        opacity: 1,
        y: 0,
        transition: { duration: 0.6, ease: "easeOut" },
    },
};

const cardAnim: Variants = {
    hidden: { opacity: 0, y: 30, scale: 0.96 },
    show: {
        opacity: 1,
        y: 0,
        scale: 1,
        transition: { duration: 0.45, ease: "easeOut" },
    },
};

const staggerCards: Variants = {
    hidden: {},
    show: {
        transition: { staggerChildren: 0.15 },
    },
};

interface TestimonialItem {
    name: string;
    designation: string;
    description: string;
    image: string;
    location?: string;
    rating?: string;
}

interface TestimonialsData {
    headline: string;
    description: string;
    testimonials: TestimonialItem[];
}

interface TestimonialsSectionProps {
    data: TestimonialsData | null;
}

function stripHtml(html: string): string {
    if (!html) return "";
    return html.replace(/<[^>]*>?/gm, "").trim();
}

export default function TestimonialsSection({ data }: TestimonialsSectionProps) {
    if (!data || !data.testimonials?.length) {
        return null;
    }

    const { headline, description, testimonials } = data;

    return (
        <section className="brs-wrapper py-5">
            <motion.div
                className="container text-center"
                initial="hidden"
                whileInView="show"
                viewport={{ once: true, amount: 0.2 }}
            >
                {/* Header */}
                <motion.div variants={fadeUp}>
                    <h2 className="brs-heading mb-2">
                        {headline}
                    </h2>

                    <div
                        className="brs-subtext mb-4"
                        dangerouslySetInnerHTML={{ __html: stripHtml(description) }}
                    />
                </motion.div>

                {/* Reviews Slider */}
                <motion.div
                    className="brs-reviews"
                    initial="hidden"
                    whileInView="show"
                    viewport={{ once: true, amount: 0.2 }}
                >
                    <motion.div variants={staggerCards}>
                        <Swiper
                            modules={[Autoplay]}
                            autoplay={{ delay: 3500, disableOnInteraction: false }}
                            loop={testimonials.length > 3}
                            spaceBetween={24}
                            slidesPerView={3}
                            breakpoints={{
                                0: { slidesPerView: 1, spaceBetween: 16 },
                                576: { slidesPerView: 1.3, spaceBetween: 18 },
                                768: { slidesPerView: 2, spaceBetween: 20 },
                                1200: { slidesPerView: 3, spaceBetween: 24 },
                            }}
                        >
                            {testimonials.map((t, idx) => (
                                <SwiperSlide key={idx}>
                                    <motion.div
                                        variants={cardAnim}
                                        className="d-flex h-100 justify-content-center"
                                    >
                                        <div className="brs-card rounded-4 p-4 flex-grow-1 position-relative text-start">
                                            <p className="brs-card-title mb-2 mt-4">
                                                {t.designation}
                                            </p>

                                            <div
                                                className="brs-card-text mb-3"
                                                dangerouslySetInnerHTML={{
                                                    __html: stripHtml(t.description),
                                                }}
                                            />

                                            <div className="d-flex gap-3">
                                                <img
                                                    src={t.image}
                                                    alt={t.name}
                                                    className="img-fluid"
                                                    style={{
                                                        width: 50,
                                                        height: 50,
                                                        borderRadius: "50%",
                                                    }}
                                                />

                                                <div>
                                                    <p className="mb-1 brs-card-name">
                                                        {t.name}
                                                    </p>

                                                    {t.location && (
                                                        <div
                                                            className="mb-0 brs-card-location"
                                                            dangerouslySetInnerHTML={{
                                                                __html: stripHtml(t.location),
                                                            }}
                                                        />
                                                    )}

                                                    {t.rating && (
                                                        <div className="brs-card-rating mb-0">
                                                            <span className="text-warning">
                                                                {t.rating}
                                                            </span>
                                                        </div>
                                                    )}
                                                </div>
                                            </div>

                                            <div className="brs-card-quote">
                                                <i className="fa-solid fa-quote-left"></i>
                                            </div>
                                        </div>
                                    </motion.div>
                                </SwiperSlide>
                            ))}
                        </Swiper>
                    </motion.div>
                </motion.div>
            </motion.div>
        </section>
    );
}
