"use client";

import React, { createContext, useContext, useEffect, useState, useCallback } from "react";
import { API_ENDPOINTS, getToken } from "@/lib/apiEndpoints";

/* -----------------------------
    Types
------------------------------*/
export type CartItem = {
    id: string;
    name: string;
    price: number;
    qty: number;
    img: string;
    pageName: string;
    subjectIds: string[] | null;

    courseCode: string;
    itemCode: string;
    selectedType: string | null;
    selectedMedium: string | null;
    selectedSession: string | null;
    dataSaveId: string | null;
};

type CartContextType = {
    items: CartItem[];
    isOpen: boolean;
    isLoading: boolean;

    openCart: () => void;
    closeCart: () => void;
    toggleCart: () => void;

    addToCart: (item: CartItem) => Promise<void>;
    removeFromCart: (id: string) => Promise<void>;
    changeQty: (id: string, delta: number) => Promise<void>;
    clearCart: () => Promise<void>;

    setBuyNowItem: (item: CartItem) => void;
    buyNowItem: CartItem | null;
    clearBuyNowItem: () => void;

    mergeGuestCartToUser: () => Promise<void>;
    loadCart: () => Promise<void>;

    cartTotal: number;
    cartCount: number;
};

const CartContext = createContext<CartContextType | undefined>(undefined);

/* -----------------------------
    Provider
------------------------------*/
export function CartProvider({ children }: { children: React.ReactNode }) {
    const [items, setItems] = useState<CartItem[]>([]);
    const [isOpen, setIsOpen] = useState(false);
    const [isLoading, setIsLoading] = useState(true);
    const [buyNowItem, setBuyNowItem] = useState<CartItem | null>(null);

    const cartChannel =
        typeof window !== "undefined" ? new BroadcastChannel("cart-sync") : null;


    const isLoggedIn = () => !!getToken();

    /* -----------------------------
        Load cart on mount
    ------------------------------*/
    const loadCart = useCallback(async () => {
        setIsLoading(true);

        try {
            // GUEST
            if (!isLoggedIn()) {
                const saved = localStorage.getItem("cart_items");
                if (saved) {
                    setItems(JSON.parse(saved));
                } else {
                    setItems([]);
                }
                return;
            }

            // LOGGED-IN
            const res = await API_ENDPOINTS.CART.LIST();
            if (res.ok) {
                const data = await res.json();
                setItems(data.items || []);
            }
        } catch (err) {
            console.error("Load cart failed", err);
            setItems([]);
        } finally {
            setIsLoading(false);
        }
    }, []);

    /* -----------------------------
        Initial load
    ------------------------------*/
    useEffect(() => {
        loadCart();
    }, [loadCart]);

    /* -----------------------------
        Persist guest cart only
    ------------------------------*/
    useEffect(() => {
        if (!isLoggedIn()) {
            localStorage.setItem("cart_items", JSON.stringify(items));
        }
    }, [items]);

    /* -----------------------------
        Drawer controls
    ------------------------------*/
    const openCart = () => setIsOpen(true);
    const closeCart = () => setIsOpen(false);
    const toggleCart = () => setIsOpen((p) => !p);
    let isMergingCart = false;

    /* -----------------------------
        Merge guest cart after login
    ------------------------------*/
    const mergeGuestCartToUser = async () => {
        if (isMergingCart) return;
        if (!isLoggedIn()) return;

        const saved = localStorage.getItem("cart_items");
        if (!saved) return;

        const guestCart: CartItem[] = JSON.parse(saved);
        if (!guestCart.length) return;

        isMergingCart = true;
        localStorage.removeItem("cart_items");

        try {
            const res = await API_ENDPOINTS.CART.MERGE(guestCart);

            if (!res?.ok) {
                console.error("Cart merge failed");
                return;
            }

            await loadCart();
        } finally {
            isMergingCart = false;
        }
    };

    /* -----------------------------
        React to auth changes
    ------------------------------*/
    useEffect(() => {
        const onLogin = async () => {
            await mergeGuestCartToUser();
            await loadCart();
        };

        const onLogout = () => {
            loadCart();
        };

        window.addEventListener("auth-login", onLogin);
        window.addEventListener("auth-logout", onLogout);

        return () => {
            window.removeEventListener("auth-login", onLogin);
            window.removeEventListener("auth-logout", onLogout);
        };
    }, [loadCart]);

    /* -----------------------------
        Add to cart
    ------------------------------*/
    const addToCart = async (newItem: CartItem) => {
        // GUEST
        if (!isLoggedIn()) {
            setItems((prev) => {
                const existing = prev.find((i) => i.id === newItem.id);
                if (existing) {
                    return prev.map((i) =>
                        i.id === newItem.id
                            ? { ...i, qty: i.qty + newItem.qty }
                            : i
                    );
                }
                return [...prev, newItem];
            });

            cartChannel?.postMessage("updated");
            openCart();
            return;
        }

        // LOGGED-IN
        const res = await API_ENDPOINTS.CART.ADD(newItem);
        if (!res.ok) {
            console.error("Add to cart failed");
            return;
        }

        // setItems((prev) => {
        //     const existing = prev.find((i) => i.id === newItem.id);
        //     if (existing) {
        //         return prev.map((i) =>
        //             i.id === newItem.id
        //                 ? { ...i, qty: i.qty + newItem.qty }
        //                 : i
        //         );
        //     }
        //     return [...prev, newItem];
        // });

        await loadCart();
        cartChannel?.postMessage("updated");
        openCart();
    };

    /* -----------------------------
        Remove item
    ------------------------------*/
    const removeFromCart = async (id: string) => {
        if (!isLoggedIn()) {
            setItems((prev) => prev.filter((i) => i.id !== id));
            cartChannel?.postMessage("updated");
            return;
        }

        const res = await API_ENDPOINTS.CART.REMOVE(id);
        if (!res.ok) {
            console.error("Remove failed");
            return;
        }

        setItems((prev) => prev.filter((i) => i.id !== id));
        cartChannel?.postMessage("updated");
    };

    /* -----------------------------
        Change quantity
    ------------------------------*/
    const changeQty = async (id: string, delta: number) => {
        const item = items.find((i) => i.id === id);
        if (!item) return;

        const newQty = item.qty + delta;
        if (newQty <= 0) return;

        if (!isLoggedIn()) {
            setItems((prev) =>
                prev.map((i) =>
                    i.id === id ? { ...i, qty: newQty } : i
                )
            );

            cartChannel?.postMessage("updated");
            return;
        }

        const res = await API_ENDPOINTS.CART.UPDATE(id, newQty);
        if (!res.ok) {
            console.error("Update qty failed");
            return;
        }

        setItems((prev) =>
            prev.map((i) =>
                i.id === id ? { ...i, qty: newQty } : i
            )
        );

        cartChannel?.postMessage("updated");
    };

    /* -----------------------------
        Clear cart
    ------------------------------*/
    const clearCart = async () => {
        if (!isLoggedIn()) {
            setItems([]);
            localStorage.removeItem("cart_items");
            cartChannel?.postMessage("updated");
            return;
        }

        const res = await API_ENDPOINTS.CART.CLEAR();
        if (!res.ok) {
            console.error("Clear cart failed");
            return;
        }

        setItems([]);
        cartChannel?.postMessage("updated");
    };

    /* -----------------------------
        Clear Buy Now
    ------------------------------*/
    const clearBuyNowItem = () => {
        setBuyNowItem(null);
    };

    /* -----------------------------
        Listener
    ------------------------------*/
    useEffect(() => {
        if (!cartChannel) return;

        const handler = async () => {
            await loadCart();
        };

        cartChannel.addEventListener("message", handler);

        return () => {
            cartChannel.removeEventListener("message", handler);
        };
    }, [loadCart]);

    /* -----------------------------
        Derived values
    ------------------------------*/
    const cartTotal = items.reduce((acc, i) => acc + i.price * i.qty, 0);
    const cartCount = items.reduce((acc, i) => acc + i.qty, 0);

    return (
        <CartContext.Provider
            value={{
                items,
                isOpen,
                isLoading,

                openCart,
                closeCart,
                toggleCart,

                addToCart,
                removeFromCart,
                changeQty,
                clearCart,

                buyNowItem,
                setBuyNowItem,
                clearBuyNowItem,

                mergeGuestCartToUser,
                loadCart,

                cartTotal,
                cartCount,
            }}
        >
            {children}
        </CartContext.Provider>
    );
}

/* -----------------------------
    Hook
------------------------------*/
export function useCart() {
    const ctx = useContext(CartContext);
    if (!ctx) {
        throw new Error("useCart must be used within CartProvider");
    }
    return ctx;
}
