import { Metadata } from "next";

function stripHtml(html: string): string {
    if (!html) return "";
    return html.replace(/<[^>]*>?/gm, "").trim();
}

/**
 * Metadata for the main Assignment Category Listing Page
 * e.g., /assignments/solved, /assignments/handwritten
 */
export function buildAssignmentListMeta(data: any, defaultTitle: string = "Assignments"): Metadata {
    const title = data.meta_title || data.page_headline || defaultTitle;
    const description = stripHtml(data.meta_description || data.page_description || "");
    const keywords = data.meta_keywords || "";

    return {
        title,
        description,
        keywords,
        openGraph: {
            title,
            description,
            type: "website",
        },
        twitter: {
            card: "summary_large_image",
            title,
            description,
        },
        // Canonical will be handled in the page.tsx based on the route
    };
}

/**
 * Metadata for the Course/Subcategory Page
 * e.g., /assignments/solved/bca
 */
export function buildAssignmentCourseMeta(course: any): Metadata {
    const title = course.meta_title || course.course_headline || course.title;
    const description = stripHtml(course.meta_description || course.description || "");
    const keywords = course.meta_keywords || "";

    return {
        title,
        description,
        keywords,
        openGraph: {
            title,
            description,
            type: "article",
        },
        twitter: {
            card: "summary_large_image",
            title,
            description,
        },
        // Canonical to be set in page
    };
}

/**
 * Metadata for the Assignment Detail Page
 * e.g., /assignments/solved/bca/mcs-011
 */
export function buildAssignmentDetailMeta(assignment: any): Metadata {
    const title = assignment.meta_title || assignment.title;
    const description = stripHtml(assignment.meta_description || assignment.short_description || assignment.main_description || "");
    const keywords = assignment.meta_keywords || "";

    // Extract tags from keywords if needed
    const tags = keywords ? keywords.split(",").map((t: string) => t.trim()) : [];

    const images = assignment.image ? [
        {
            url: assignment.image,
            width: 1200,
            height: 630,
            alt: title,
        },
    ] : undefined;

    return {
        title,
        description,
        keywords,
        openGraph: {
            title,
            description,
            type: "article",
            images: images,
            tags: tags.length > 0 ? tags : undefined,
        },
        twitter: {
            card: "summary_large_image",
            title,
            description,
            images: images,
        },
        // Canonical to be set in page
    };
}
