import { Metadata } from "next";

function stripHtml(html: string): string {
    if (!html) return "";
    return html.replace(/<[^>]*>?/gm, "").trim();
}

/**
 * Metadata for Guess Papers Main Listing /guess-papers
 */
export function buildGuessPaperListMeta(data: any): Metadata {
    const title = data.meta_title || "IGNOU Guess Papers | Excelsius";
    const description = stripHtml(data.meta_description || "Get the latest IGNOU Guess Papers to score high in your exams.");
    const keywords = data.meta_keywords || "ignou guess papers, ignou important questions, ignou exam preparation";

    return {
        title,
        description,
        keywords,
        openGraph: {
            title,
            description,
            type: "website",
        },
        twitter: {
            card: "summary_large_image",
            title,
            description,
        },
        alternates: {
            canonical: "/guess-papers",
        },
    };
}

/**
 * Metadata for Guess Papers Category (Course) /guess-papers/[slug]
 */
export function buildGuessPaperCategoryMeta(course: any, slug: string): Metadata {
    const title = course.meta_title_guess_papers || course.course_headline || `IGNOU ${course.code || "Guess Papers"}`;
    const description = stripHtml(course.meta_description_guess_papers || course.description || `Guess Papers for ${course.code}`);
    const keywords = course.meta_keyword_guess_papers || "";

    return {
        title,
        description,
        keywords,
        openGraph: {
            title,
            description,
            type: "article",
        },
        twitter: {
            card: "summary_large_image",
            title,
            description,
        },
        alternates: {
            canonical: `/guess-papers/${slug}`,
        },
    };
}

/**
 * Metadata for Guess Paper Detail /guess-papers/[slug]/[code]
 */
export function buildGuessPaperDetailMeta(paper: any, courseSlug: string, paperSlug: string): Metadata {
    const title = paper.meta_title || paper.title || paper.code;
    const description = stripHtml(paper.meta_description || paper.short_description || paper.main_description || "");
    const keywords = paper.meta_keywords || "";

    // Extract images if available
    const images = paper.image ? [
        {
            url: paper.image,
            width: 1200,
            height: 630,
            alt: title,
        },
    ] : undefined;

    return {
        title,
        description,
        keywords,
        openGraph: {
            title,
            description,
            type: "article",
            images,
        },
        twitter: {
            card: "summary_large_image",
            title,
            description,
            images,
        },
        alternates: {
            canonical: `/guess-papers/${courseSlug}/${paperSlug}`,
        },
    };
}
