import { Metadata } from "next";

function stripHtml(html: string): string {
    if (!html) return "";
    return html.replace(/<[^>]*>?/gm, "").trim();
}

/**
 * Metadata for Project Listing /projects
 */
export function buildProjectListMeta(data: any): Metadata {
    const title = data.meta_title || data.page_headline || "IGNOU Projects";
    const description = stripHtml(data.meta_description || data.page_description || "Browse our collection of IGNOU projects.");
    const keywords = data.meta_keywords || "";

    return {
        title,
        description,
        keywords,
        openGraph: {
            title,
            description,
            type: "website",
        },
        twitter: {
            card: "summary_large_image",
            title,
            description,
        },
        alternates: {
            canonical: "/projects",
        },
    };
}

/**
 * Metadata for Course Projects Listing /projects/[code]
 */
export function buildCourseMeta(course: any): Metadata {
    const title = course.meta_title_projects || course.course_headline || course.title || "Course Projects";
    const description = stripHtml(course.meta_description_projects || course.description || `Projects for ${course.code}`);
    const keywords = course.meta_keyword_projects || "";

    return {
        title,
        description,
        keywords,
        openGraph: {
            title,
            description,
            type: "article",
        },
        twitter: {
            card: "summary_large_image",
            title,
            description,
        },
        alternates: {
            canonical: `/projects/${course.course_slug || course.enc_id}`,
        },
    };
}

/**
 * Metadata for Project Detail /projects/[code]/[projectId]
 */
export function buildProjectDetailMeta(project: any, courseCode: string): Metadata {
    const title = project.meta_title || project.title;
    const description = stripHtml(project.meta_description || project.short_description || project.main_description || "");
    const keywords = project.meta_keywords || "";
    const tags = keywords ? keywords.split(",").map((t: string) => t.trim()) : [];

    const images = project.image ? [
        {
            url: project.image,
            width: 1200,
            height: 630,
            alt: title,
        },
    ] : undefined;

    return {
        title,
        description,
        keywords,
        openGraph: {
            title,
            description,
            type: "article",
            images: images,
            tags: tags,
        },
        twitter: {
            card: "summary_large_image",
            title,
            description,
            images: images,
        },
        alternates: {
            canonical: `/projects/${courseCode}/${project.project_slug}`,
        },
    };
}
